/*
 *  Copyright (C) 2004/2005 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.HashMap;

import javax.swing.JComboBox;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.items.mischungen.MischungsLagerItem;
import de.lunqual.rzpro.items.tank.ortItem;
import de.lunqual.rzpro.items.tank.ortListe;
import de.lunqual.rzpro.items.tank.tankInhaltsItem;
import de.lunqual.rzpro.items.tank.tankInhaltsItemExt;
import de.lunqual.rzpro.items.tank.tankItem;
import de.lunqual.rzpro.items.tank.tankItemListe;
import de.lunqual.rzpro.log.LogFactory;

/**
 *
 * @author  lunqual
 */
public class DBTank{

    RzPro           		rz;
    Connection		      	con;
    DBFactory       		db;

    PreparedStatement   	getOrte;
    PreparedStatement		getOrt;
    PreparedStatement		getTankAlle;
    PreparedStatement		getTankOrt;
    PreparedStatement		getTank;
    PreparedStatement		saveOrt;
    PreparedStatement		updateOrt;
    PreparedStatement		deleteOrt;
    PreparedStatement		checkOrtDelete;
    PreparedStatement		saveTank;
    PreparedStatement		updateTank;
    PreparedStatement		tankExists;
    PreparedStatement		deleteTank;
    PreparedStatement		getTankRegex;
    PreparedStatement		getTankNummer;
    PreparedStatement		getTankID;
    PreparedStatement		getTankFuellung;
    PreparedStatement		getTankFromLosnummer;
    PreparedStatement		getTankFromSorte;
    PreparedStatement		getTankLocked;
    PreparedStatement		getTanklisteLocked;
    PreparedStatement		getTanklisteFuellung;		
    PreparedStatement		getTankSorteFromMischung;
    PreparedStatement		getMischungLager;

    /** Creates a new instance of DBLager */
    public DBTank(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
    }

    private void setStatements(){
        try{
        	
            getOrte   = con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_ORTE + " ORDER BY bezeichnung",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getOrt    = con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_ORTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankAlle   = con.prepareStatement("SELECT tanks.liter,tanks.id,tanks.nummer,tanks.ort,tanks.px,tanks.py,tanks.pr,tanks.comment,orte.id,orte.bezeichnung,orte.comment,orte.shape FROM " +DBFactory.TABLE_TANKS + " left join " + DBFactory.TABLE_ORTE +" on tanks.ort=orte.id ORDER BY tanks.nummer",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankOrt   = con.prepareStatement("SELECT tanks.liter,tanks.id,tanks.nummer,tanks.ort,tanks.liter,tanks.px,tanks.py,tanks.pr,tanks.comment,orte.id,orte.bezeichnung,orte.comment,orte.shape FROM " +DBFactory.TABLE_TANKS + " left join " + DBFactory.TABLE_ORTE +" on tanks.ort=orte.id  WHERE ort = ? ORDER BY tanks.nummer",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTank   = con.prepareStatement("SELECT tanks.liter,tanks.id,tanks.nummer,tanks.ort,tanks.px,tanks.py,tanks.pr,tanks.comment,orte.id,orte.bezeichnung,orte.comment,orte.shape FROM " +DBFactory.TABLE_TANKS + " left join " + DBFactory.TABLE_ORTE +" on tanks.ort=orte.id WHERE tank.id=? ORDER BY tanks.nummer",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveOrt   = con.prepareStatement("INSERT INTO orte (bezeichnung,comment,shape) VALUES (?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateOrt = con.prepareStatement("UPDATE orte set bezeichnung=?,comment=?,shape=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            checkOrtDelete   = con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_TANKS + " WHERE ort=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            deleteOrt = con.prepareStatement("DELETE FROM orte WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            saveTank = con.prepareStatement("INSERT INTO tanks (nummer,liter,comment,px,py,pr,ort) VALUES (?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateTank = con.prepareStatement("UPDATE tanks set nummer=?,liter=?,comment=?,px=?,py=?,pr=?,ort=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            tankExists   = con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_TANKS+ " WHERE nummer=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            deleteTank = con.prepareStatement("DELETE FROM tanks WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getTankRegex   = con.prepareStatement("SELECT tanks.liter,tanks.id,tanks.nummer,tanks.ort,tanks.px,tanks.py,tanks.pr,tanks.comment,orte.id,orte.bezeichnung,orte.comment,orte.shape FROM " +DBFactory.TABLE_TANKS + " left join " + DBFactory.TABLE_ORTE +" on tanks.ort=orte.id WHERE tanks.nummer regexp ? ORDER BY tanks.nummer",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankNummer   = con.prepareStatement("SELECT tanks.liter,tanks.id,tanks.nummer,tanks.ort,tanks.px,tanks.py,tanks.pr,tanks.comment,orte.id,orte.bezeichnung,orte.comment,orte.shape FROM " +DBFactory.TABLE_TANKS + " left join " + DBFactory.TABLE_ORTE +" on tanks.ort=orte.id WHERE tanks.nummer=? ORDER BY tanks.nummer",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankID   = con.prepareStatement("SELECT tanks.liter,tanks.id,tanks.nummer,tanks.ort,tanks.px,tanks.py,tanks.pr,tanks.comment,orte.id,orte.bezeichnung,orte.comment,orte.shape FROM " +DBFactory.TABLE_TANKS + " left join " + DBFactory.TABLE_ORTE +" on tanks.ort=orte.id WHERE tanks.id=? ORDER BY tanks.nummer",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankFuellung   = con.prepareStatement("SELECT lager.name,lager.hg,lager.vg,buchungen.losnummer,buchungen.datum,bestand,meldebestand, buchungen.bezeichnung,rezeptur_id, rest_liter as restliter,rest_kg FROM buchungen left join rezeptliste on buchungen.rezeptur_id=rezeptliste.id left join lager on buchungen.lager=lager.id where rest <> 0 AND buchungen.comment regexp ? order by buchungen.id asc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankFromLosnummer   = con.prepareStatement("SELECT comment FROM buchungen WHERE buchungen.rest != 0 AND buchungen.losnummer = ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankFromSorte   = con.prepareStatement("SELECT comment FROM buchungen WHERE buchungen.rest != 0 AND buchungen.bezeichnung regexp ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTankSorteFromMischung   = con.prepareStatement("SELECT id,bezeichnung,tank FROM mischungen WHERE active != 0 and bezeichnung regexp ?  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
                      
            getTankLocked   = con.prepareStatement("SELECT * FROM mischungen WHERE tank=? and active != 0 ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getTanklisteLocked   = con.prepareStatement("SELECT * FROM mischungen WHERE active != 0 ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            
//            getTanklisteFuellung   = con.prepareStatement("SELECT lager.name,lager.hg,lager.vg,buchungen.comment,buchungen.losnummer,buchungen.datum,bestand,meldebestand, buchungen.bezeichnung,rezeptur_id, rest_liter as restliter,rest_kg FROM buchungen left join rezeptliste on buchungen.rezeptur_id=rezeptliste.id left join lager on buchungen.lager=lager.id where rest <> 0 AND buchungen.comment regexp '[[.left-square-bracket.]]Tank.*[[:<:]].*[[:>:]].*[[.right-square-bracket.]]' order by buchungen.id asc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
//            getTanklisteFuellung   = con.prepareStatement("SELECT lager.name,lager.hg,lager.vg,buchungen.comment,buchungen.losnummer,buchungen.datum,bestand,meldebestand, buchungen.bezeichnung,rezeptur_id, rest_liter as restliter,rest_kg FROM buchungen left join rezeptliste on buchungen.rezeptur_id=rezeptliste.id left join lager on buchungen.lager=lager.id where rest <> 0 AND buchungen.comment regexp '\\[Tank.*\\]' order by buchungen.id asc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
 
            getTanklisteFuellung   = con.prepareStatement("SELECT lager.name,lager.hg,lager.vg,buchungen.comment,buchungen.losnummer,buchungen.datum,bestand,meldebestand, buchungen.bezeichnung,rezeptur_id, rest_liter as restliter,rest_kg FROM buchungen left join rezeptliste on buchungen.rezeptur_id=rezeptliste.id left join lager on buchungen.lager=lager.id where rest <> 0 AND buchungen.comment regexp '\\\\[Tank.*[[:<:]].*[[:>:]].*\\\\]' order by buchungen.id asc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            
            getMischungLager   = con.prepareStatement("SELECT mischungen.id,mischungen.lager,lager.id,lager.name,lager.hg,lager.vg FROM mischungen left join lager on mischungen.lager=lager.id where mischungen.id=?  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTanksetStatements", e.getLocalizedMessage());
        }
    }

    public MischungsLagerItem getMischungsLagerItem(int id) {
    	MischungsLagerItem ret = null;
    	try {
    		if(id != 0) {
	    		getMischungLager.setInt(1, id);
	    		final ResultSet rs = getMischungLager.executeQuery();
	    		if(rs.first()) {
	    			ret = new MischungsLagerItem(
	    					rs.getInt("lager.id"),
	    					rs.getString("lager.name"),
	    					rs.getString("lager.hg"),
	    					rs.getString("lager.vg")
	    					);
	    		}
    		}
		}catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getMischungsLagerItem", e.getLocalizedMessage());
	    }
		
		return ret;
	}
    public String getTankString(String c) {
    	String ret = "";
    	if(c.matches(".*\\[Tank.*\\].*")) {
			String tank_anfang_token=rz.getOptionFactory().getOption("dbbuchung.tank_sb");
            String tank_ende_token=rz.getOptionFactory().getOption("dbbuchung.tank_se");
            String tanks="";
        	int ta = c.indexOf(tank_anfang_token);
        	if(ta != -1) {
        		String t1 = c.substring(ta);
            	int te = t1.indexOf(tank_ende_token);
            	if(te != -1) {
            		tanks = c.substring(ta+tank_anfang_token.length(),ta+te);
            		ret += tanks.replace(":", " ")+" ";
            	}
        	}
		}
    	return ret;
    }
    public String getTankFromSorte(String sorte) {
    	String ret ="";
    	try {
    		getTankFromSorte.setString(1, sorte);
    		ResultSet rs = getTankFromSorte.executeQuery();
    		while(rs.next()) {
    			String c = rs.getString("comment");
    			if(c.matches("\\[Tank.*\\]")) {
    				String tank_anfang_token=rz.getOptionFactory().getOption("dbbuchung.tank_sb");
    	            String tank_ende_token=rz.getOptionFactory().getOption("dbbuchung.tank_se");
    	            String tanks="";
    	        	int ta = c.indexOf(tank_anfang_token);
    	        	if(ta != -1) {
    	        		String t1 = c.substring(ta);
    	            	int te = t1.indexOf(tank_ende_token);
    	            	if(te != -1) {
    	            		tanks = c.substring(ta+tank_anfang_token.length(),ta+te);
    	            		ret += tanks.replaceAll(":","")+",";
    	            	}
    	        	}
    			}
    		}
    		getTankSorteFromMischung.setString(1, sorte);
    		rs = getTankSorteFromMischung.executeQuery();
    		while(rs.next()) {
    			String c = rs.getString("tank");
    			if(!c.trim().equals("")) {
    				ret += c.trim() + ",";
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankFromSorte", e.getLocalizedMessage());
        }
    	if(ret.endsWith(","))ret = ret.substring(0, ret.length() - 1);
    	return ret;
    }
    
    public String getTankFromLosnummer(String losnummer) {
    	String ret ="";
    	try {
    		getTankFromLosnummer.setString(1, losnummer);
    		final ResultSet rs = getTankFromLosnummer.executeQuery();
    		while(rs.next()) {
    			String c = rs.getString("comment");
    			if(c.matches(".*\\[Tank.*\\].*")) {
    				String tank_anfang_token=rz.getOptionFactory().getOption("dbbuchung.tank_sb");
    	            String tank_ende_token=rz.getOptionFactory().getOption("dbbuchung.tank_se");
    	            String tanks="";
    	        	int ta = c.indexOf(tank_anfang_token);
    	        	if(ta != -1) {
    	        		String t1 = c.substring(ta);
    	            	int te = t1.indexOf(tank_ende_token);
    	            	if(te != -1) {
    	            		tanks = c.substring(ta+tank_anfang_token.length(),ta+te);
    	            		ret += tanks.replaceAll(":","")+",";
    	            	}
    	        	}
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankFromLosnummer", e.getLocalizedMessage());
        }
    	if(ret.endsWith(","))ret = ret.substring(0, ret.length() - 1);
    	return ret;
    }
    
    public ArrayList<tankItem> getTankListeFromTankPattern(String in){
    	ArrayList<tankItem> ret = new ArrayList<tankItem>();
    	String input[] = in.split("\\s|,|;|:|-|\\(|\\)");
		for(String item:input) {
			try {
				ret.add(getTank(item));
			}catch (Exception e) {}
		}
    	return ret;
    }
    
    
    public ArrayList<tankItem> getTankListeFromLosnummerPattern(String in){
    	ArrayList<tankItem> ret = new ArrayList<tankItem>();
		
    	return ret;
    }
    
    public void deleteTank(tankItem ti) {
    	try {
    		deleteTank.setInt(1, ti.getId());
    		deleteTank.executeUpdate();
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.deleteTank", e.getLocalizedMessage());
        }
    }
    
    public boolean tankExists(tankItem ti) {
    	boolean ret = true;
    	try {
    		tankExists.setString(1, ti.getNummer().trim());
    		final ResultSet rs = tankExists.executeQuery();
    		if(rs.next()) {
    			ret = true;
    		}else {
    			ret = false;
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.tankexists_ti", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public boolean tankExists(String nummer) {
    	String nummern[] = nummer.split("\\s|,|;|:|-");
    	try {
    		for(String n:nummern) {
	    		tankExists.setString(1, n.trim());
	    		final ResultSet rs = tankExists.executeQuery();
	    		if(rs.next()) {
	    			return true;
	    		}else {
	    			return false;
	    		}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.tankExists_string", e.getLocalizedMessage());
        }
    	return false;
    }
    
    
    public int saveTank(tankItem ti) {
    	int ret = 0;
    	try {
    		PreparedStatement stm;
    		if(ti.getId() == 0) {
    			stm=saveTank;
    		} else {
    			stm= updateTank;
    			stm.setInt(8, ti.getId());
    		}
    		stm.setString(1, ti.getNummer());
    		stm.setDouble(2, ti.getLiter());
    		stm.setString(3, ti.getComment());
    		stm.setInt(4, ti.getPx());
    		stm.setInt(5, ti.getPy());
    		stm.setInt(6, ti.getPr());
    		stm.setInt(7, ti.getOrt());
    		stm.executeUpdate();
    		if(ti.getId()==0) {
    			ret=db.dbGetLastInsertID(DBFactory.TABLE_TANKS);
    			ti.setId(ret);
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.saveTank", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public boolean ortIsDeleteable(ortItem oi) {
    	boolean ret = false;
    	try {
    		if(oi.getId() != 0) {
    			checkOrtDelete.setInt(1, oi.getId());
    			final ResultSet rs = checkOrtDelete.executeQuery();
    			if(rs.next()) {
    				ret = false;
    			}else {
    				ret = true;
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.checkDeleteOrt", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public boolean deleteOrt(ortItem oi) {
    	boolean ret = true;
    	try {
    		if(oi.getId() != 0) {
    			ret = ortIsDeleteable(oi);
    			if(ret) {
    				deleteOrt.setInt(1, oi.getId());
    				deleteOrt.executeUpdate();
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.DeleteOrt", e.getLocalizedMessage());
    	}
    	
    	return ret;
    }
    
    public int saveOrt(ortItem oi) {
    	int ret = 0;
    	try {
    		PreparedStatement stm;
    		if(oi.getId()==0) {
    			stm=saveOrt;
    		} else {
    			stm=updateOrt;
    			stm.setInt(4, oi.getId());
    		}
    		stm.setString(1, oi.getBezeichnung());
    		stm.setString(2, oi.getComment());
    		stm.setString(3, oi.getShape());
    		stm.executeUpdate();
    		if(oi.getId()==0) {
    			ret=db.dbGetLastInsertID(DBFactory.TABLE_ORTE);
    			oi.setId(ret);
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.saveOrt", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public ortItem getOrt(int id){
    	ortItem ret = null;
    	try {
    		if(id != 0) {
	    		final ResultSet rs;
	    		getOrt.setInt(1, id);
	    		rs = getOrt.executeQuery();
	    		if(rs.next()) {
	    			ret = getOrtFromRs(rs);
	    		}
    		} else {
    			ret = new ortItem(rz,0,"","","");
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getOrt", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public void fillOrtCmb(JComboBox cmb) {
    	cmb.removeAllItems();
    	ArrayList<ortItem> liste = getOrte();
    	for(ortItem ort:liste) {
    		cmb.addItem(ort);
    	}
    }

    public void selectOrteAuswahl(ArrayList<ortItem> orte,ArrayList<tankItem> tanks) {
    	for(ortItem ort:orte) {
    		ArrayList<tankItem> liste = ort.getTanks();
    		for(tankItem tank:tanks) {
    			if(tank != null) {
	    			if(tankIsInOrt(liste,tank.getId())) {
	    				ort.setSelected(true);
	    			}
    			}
    		}
    	}
    }
    
    public void selectOrte(ArrayList<ortItem> orte,ArrayList<tankItem> tanks) {
    	for(ortItem ort:orte) {
    		ArrayList<tankItem> liste = getTanks(ort.getId());
    		for(tankItem tank:tanks) {
    			if(tank != null) {
	    			if(tankIsInOrt(liste,tank.getId())) {
	    				ort.setSelected(true);
	    			}
    			}
    		}
    	}
    }
    
    private boolean tankIsInOrt(ArrayList<tankItem> liste,int tank) {
    	boolean ret =false;
    	for(tankItem t:liste) {
    		if(t.getId()==tank) {
    			ret =true;
    			break;
    		}
    	}
    	return ret;
    }
    
    public ArrayList<ortItem> getOrte(){
    	ArrayList<ortItem> ret = new ArrayList<ortItem>();
    	try {
    		final ResultSet rs;
    		rs = getOrte.executeQuery();
    		while(rs.next()) {
    			ret.add(getOrtFromRs(rs));
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getOrte", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    private ortItem getOrtFromRs(ResultSet rs) {
    	ortItem ret = null;
    	try {
    		ret = new ortItem(rz,
    				rs.getInt("id"),
    				rs.getString("bezeichnung"),
    				rs.getString("comment"),
    				rs.getString("shape")
    				);
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getOrtFromRs", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public tankItemListe getTanksRegex(String nummer){
    	tankItemListe ret = new tankItemListe();
    	if(!nummer.trim().equals("")) {
	    	try {
	    		getTankRegex.setString(1, nummer);
	    		final ResultSet rs = getTankRegex.executeQuery();
	    		while (rs.next()) {
	    			tankItem ti = getTankFromRs(rs,true);
	    			ret.add(ti);
	    		}
			}catch (final Exception e){
		        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankRegex", e.getLocalizedMessage());
		    }
    	}
    	return ret;
    }
    
    public ortListe getAlleTanksUndOrte() {
    	ortListe ret = new ortListe(rz);
    	try {
        	ArrayList<ortItem> orte = rz.getDatabase().getTank().getOrte();
        	for(ortItem oi:orte) {
        		ret.add(oi.clones());
        	}
        	ArrayList<tankItem> tanks = getTanks(0);
        	for(tankItem ti:tanks) {
        		ret.getOrt(ti.getOrt()).getTanks().add(ti.clones());
        	}
	    }catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getAlleTanksUndOrte", e.getLocalizedMessage());
	        e.printStackTrace();
	    }
    	return ret;
    }
    
    public ArrayList<tankItem> getTanks(int ort){
    	ArrayList<tankItem> ret = new ArrayList<tankItem>();
    	try {
    		PreparedStatement stm;
    		if(ort == 0) {
    			stm = getTankAlle;
    		} else {
    			stm = getTankOrt;
    			stm.setInt(1, ort);
    		}
    		final ResultSet rs = stm.executeQuery();
    		while (rs.next()) {
    			ret.add(getTankFromRs(rs,false));    
    		}
    		getTanklisteFuellung(ret);
		}catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTanks", e.getLocalizedMessage());
	    }
    	return ret;
    }
    /*
     * ---------------------------------------------------------
     * füllung für alle Tanks einer Liste ermitteln
     * ---------------------------------------------------------
     * 
     */
    private void getTanklisteFuellung(ArrayList<tankItem> liste) {
    	try {
    		//alle Buchungen mit Tank-comment ermitteln
	    	HashMap<String,Boolean>  listenMap = new HashMap<String,Boolean>();
	    	HashMap<String,ArrayList<tankInhaltsItem>>  tankMap = new HashMap<String,ArrayList<tankInhaltsItem>>();
	    	HashMap<String,tankInhaltsItemExt> mischungMap = new HashMap<String,tankInhaltsItemExt>();
	    	for(tankItem ti:liste) {
	    		listenMap.put(ti.getNummer().toLowerCase(), true);
	    	}
	    	ResultSet rs = getTanklisteFuellung.executeQuery();
	    	while (rs.next()){
	    		String input[] = extractTank(rs.getString("buchungen.comment")).split("\\s|,|;|:|-");
	    		for(String item:input) {
	    			item = item.toLowerCase();
	    			try {
	    				if(!item.equals("")) {
		    				if(listenMap.containsKey(item)) {
			    				if(tankMap.containsKey(item)){
			    					tankMap.get(item).add(getTankInhaltsItemFromRS(rs));
			    				}else {
			    					tankMap.put(item, new ArrayList<tankInhaltsItem>());
			    					tankMap.get(item).add(getTankInhaltsItemFromRS(rs));
			    				}
			    			}
	    				}
		    		}catch (Exception e) {}
	    		}
	    	}
	    	rs = getTanklisteLocked.executeQuery();
	    	while(rs.next()) {
	    			MischungsLagerItem li = null;
	    			if(rz.getOptionFactory().getOption("tanklistenfenster.lager_anzeige", 0) != 0) {
	    				li = rz.getDatabase().getTank().getMischungsLagerItem(rs.getInt("id"));
	    			}
	    			mischungMap.put(rs.getString("tank").toLowerCase(),new tankInhaltsItemExt(
	    					rs.getString("tank"),new tankInhaltsItem(rs.getInt("id"),
	        					rs.getString("bezeichnung") ,
	        					String.valueOf(rs.getInt("id")),
	        					db.getDateFromString(rs.getString("erstellt")),
	        					0.0,
	        					rs.getDouble("la"),
	        					false,
	        					true,
	        					(li!=null)?li.getLagerName():"",
	        					(li!=null)?li.getHg():"ffffff",
	                			(li!=null)?li.getVg():"000000"
	        					)));
	    	}
	    	for(tankItem ti:liste) {
	    		if(tankMap.containsKey(ti.getNummer().toLowerCase())) {
	    			ArrayList<tankInhaltsItem> inhalt = tankMap.get(ti.getNummer().toLowerCase());
	    			for(tankInhaltsItem tii:inhalt) {
	    				ti.addInhalt(tii.clones());
	    			}
	    		}else {
	    			if(ti.isEmpty()) {
	    				if(mischungMap.containsKey(ti.getNummer().toLowerCase())) {
	    					ti.addInhalt(mischungMap.get(ti.getNummer().toLowerCase()).getTi().clones());
	    				}
	    			}
	    		}
	    	}
    	}catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTanklisteFuellung", e.getLocalizedMessage());
	    }
    }
    
    private tankInhaltsItem getTankInhaltsItemFromRS(ResultSet rs) {
    	tankInhaltsItem ret = null;
    	try {
    		ret = new tankInhaltsItem(rs.getInt("rezeptur_id"),
					rs.getString("bezeichnung") ,
					rs.getString("losnummer") ,
					db.getDateFromString(rs.getString("datum")),
					rs.getDouble("restliter"),
					rs.getDouble("rest_kg"),
					false,
					false,
					rs.getString("lager.name"),
					rs.getString("lager.hg"),
					rs.getString("lager.vg")
					);
    	}
    	catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankinhaltsItemFromRS", e.getLocalizedMessage());
	    }
    	return ret;
    }
    
    public String extractTank(String in) {
    	String ret = "";
    	String tank_anfang_token=rz.getOptionFactory().getOption("dbbuchung.tank_sb");
        String tank_ende_token=rz.getOptionFactory().getOption("dbbuchung.tank_se");
        int ta = in.indexOf(tank_anfang_token);
    	if(ta != -1) {
    		String t1 = in.substring(ta);
        	int te = t1.indexOf(tank_ende_token);
        	if(te != -1) {
        		ret = in.substring(ta+tank_anfang_token.length(),ta+te);
        	}
    	}
    	if(!ret.trim().equals("")) {
    		try {
    			in = in.replaceAll("\\" + tank_anfang_token+".*"+ret + "\\"+tank_ende_token, "");
    		}catch(Exception e) {}
    	}
    	return ret;
    }
    
    public tankItem getTank(int id){
    	tankItem ret = null;
    	try {
    		if(id != 0) {
	    		getTankID.setInt(1,id);
	    		final ResultSet rs = getTankID.executeQuery();
	    		if (rs.next()) {
	    			ret = getTankFromRs(rs,true);
	    		}
    		} else {
    			ret = new tankItem(rz,"");
    		}
		}catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankID", e.getLocalizedMessage());
	    }
    	return ret;
    }

    public ArrayList<tankItem> getTankInhaltsListe(String liste) {
    	tankItemListe ret = null;
    	String regex = liste.replaceAll("\\s+|,|\\.|\\:","|").replaceAll("\\|+","|");
    	ret = getTanksRegex(regex);
    	String n[] = regex.split("\\\\");
    	for(String nummer:n) {
    		tankItem ti =getTank(nummer);
    		if(ti != null) {
    			if(!ret.isTankInList(nummer)) {
    				ret.add(ti);
    			}
    		}
    	}
    	return ret;
    }
    
    public ArrayList<tankItem> getTankInhaltsListeSingle(String liste) {
    	tankItemListe ret = null;
    	String regex =liste.trim().replaceAll("\\s+","");
    	ret = getTanksRegex("^" + regex + "$");
   		tankItem ti =getTank(regex);
    		if(ti != null) {
    			if(!ret.isTankInList(regex)) {
    				ret.add(ti);
    			}
    	}
    	return ret;
    }
    
    public tankItem getTank(String nummer){
    	tankItem ret = null;
    	try {
    		if(!nummer.equals("")) {
    	    	String nummern[] = nummer.split("\\s|,|;|:|-");
    	    	for(String n:nummern) {
		    		getTankNummer.setString(1, n);
		    		final ResultSet rs = getTankNummer.executeQuery();
		    		if (rs.next()) {
		    			return getTankFromRs(rs,true);
		    		} else {
		    			//Tank ist nicht in der offiziellen Tankliste
		    			//In den Beständen nachsehen, ob in dem Tank was drin ist
		    	    	ret = new tankItem(rz,nummer);
		    	    	getTankFuellung(ret);
		    	    	ret.setStellplatz();
		    		}
    	    	}
    	    } else {
    	    	ret = new tankItem(rz,nummer);
    	    } 
		}catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankNummer", e.getLocalizedMessage());
	    }
    	return ret;
    }
    
    /*
     * SQL zur Suche [tank:nummer] im Comment-Feld
     */
    private String getTankSQLFromComment(String nummer) {
		return "\\[Tank[^[]]*[[:<:]]" +  nummer.trim() + "[[:>:]].*\\]";
    }    

    public HashMap<Integer,Integer> getRezepturenFromTankNummer(String nummer){
    	HashMap<Integer,Integer> ret = new HashMap<Integer,Integer>();
    	try {
    		String nummern[] = nummer.split(",| |;|:");
    		for(String n:nummern) {
    			if(!n.trim().equals("")) {
    				//Buchungen
					String sql = getTankSQLFromComment(n);
					getTankFuellung.setString(1, sql);
					ResultSet rs = getTankFuellung.executeQuery();
					while(rs.next()) {
						int r = rs.getInt("rezeptur_id");
						if(!ret.containsKey(r)) {
							ret.put(r, r);
						}
					}
					//Mischungen
					int r = rz.getDatabase().getMischungen().getTankRezepturID(n.trim());
					if(r!=0&&!ret.containsKey(r)) {
						ret.put(r, r);
					}
    			}
    		}
    	}catch(Exception e) {
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankRezepturenFromNummer", e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    private void getTankFuellung(tankItem ti) {
    	try {
			String sql = getTankSQLFromComment(ti.getNummer());
    		int rezeptur=0;
    		double meldebestand = 0.0;
    		double bestand = 0.0;
    		getTankFuellung.setString(1, sql);
    		ResultSet rs = getTankFuellung.executeQuery();
    		while(rs.next()) {
    			ti.addInhalt(new tankInhaltsItem(rs.getInt("rezeptur_id"),
    					rs.getString("bezeichnung") ,
    					rs.getString("losnummer") ,
    					db.getDateFromString(rs.getString("datum")),
    					rs.getDouble("restliter"),
    					rs.getDouble("rest_kg"),
    					ti.isStellplatz(),
    					false,
    					rs.getString("lager.name"),
    					rs.getString("lager.hg"),
    					rs.getString("lager.vg")
    					));
    			rezeptur = rs.getInt("rezeptur_id");
    			meldebestand = rs.getDouble("meldebestand");
    			bestand = rs.getDouble("bestand");
    		}
    		if(!rz.isZero(meldebestand)) {
    			if(bestand <= meldebestand) {
    				ti.setMeldebestand(true);
    			}else {
    				ti.setMeldebestand(false);
    			}
    		}
    		ti.setInhaltID(rezeptur);
    		if(ti.isEmpty()) {
    			//Mischung ?
    			getTankLocked.setString(1,ti.getNummer());
    			rs = getTankLocked.executeQuery();
    			MischungsLagerItem li = null;
    			if(rs.next()) {
    				if(rz.getOptionFactory().getOption("tanklistenfenster.lager_anzeige", 0) != 0) {
	    				li = rz.getDatabase().getTank().getMischungsLagerItem(rs.getInt("id"));
	    			}
    				ti.addInhalt(new tankInhaltsItem(rs.getInt("id"),
        					rs.getString("bezeichnung") ,
        					String.valueOf(rs.getInt("id")),
        					db.getDateFromString(rs.getString("erstellt")),
        					0.0,
        					rs.getDouble("la"),
        					false,
        					true,
        					(li!=null)?li.getLagerName():"",
        					(li!=null)?li.getHg():"ffffff",
        					(li!=null)?li.getVg():"000000"
        				));
    			}
    		}
    	}catch(Exception e) {
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankFuellung", e.getLocalizedMessage());
    	}
    }
    
    public tankItem getTank(int id,int ort){
    	tankItem ret = null;
    	try {
    		if(id != 0) {
	    		getTank.setInt(1, id);
	    		final ResultSet rs = getTank.executeQuery();
	    		if (rs.next()) {
	    			ret = getTankFromRs(rs,true);
	    		}
    		} else {
    			ret = new tankItem(rz,0,"","",0,0,0,50,ort, "","",false);
    		}
		}catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getTankIDORT", e.getLocalizedMessage());
	    }
    	return ret;
    }

    
    
    private tankItem getTankFromRs(ResultSet rs,boolean getFuellung) {
    	tankItem ret = null;
    	try {
    		ret = new tankItem(
    				rz,
    				rs.getInt("tanks.id"),
    				rs.getString("tanks.nummer"),
    				rs.getString("tanks.comment"),
    				rs.getDouble("tanks.liter"),
    				rs.getInt("tanks.px"),
    				rs.getInt("tanks.py"),
    				rs.getInt("tanks.pr"),
    				rs.getInt("tanks.ort"),
    				rs.getString("orte.bezeichnung"),
    				rs.getString("orte.comment"),
    				false
    				);
    		if(ret != null && getFuellung){
    			getTankFuellung(ret);
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBTank.getOrtFromRs", e.getLocalizedMessage());
        }
    	return ret;
    }
    
}
