/*
 *  Copyright (C) database.stichwort_loeschen4 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Hashtable;
import java.util.StringTokenizer;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.fenster.dialoge.DialogControl;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.items.stichworte.StichwortItem;
import de.lunqual.rzpro.items.stichworte.StichwortListe;
import de.lunqual.rzpro.log.LogFactory;
/**
 *
 * @author  lunqual
 */
public class DBSTWListe {

    RzPro           rz;
    Connection      con;
    DBFactory       db;
    DialogItem		dlgItem;
    Hashtable		HGColors;

    Statement					adressenStatement;
    
    PreparedStatement   getStichwort;
    PreparedStatement   getStichwortListe;
    PreparedStatement   getStichwortListeFilter;
    PreparedStatement   saveStichwort;
    PreparedStatement   deleteStichwort;
    PreparedStatement   getStichwortUsed;
    PreparedStatement   checkStichwort;
    PreparedStatement   updateStichwort;

    PreparedStatement		getClientfilter;
    PreparedStatement		setClientfilter;

    PreparedStatement       getAblageFilter;
    PreparedStatement       updateAblageFilter;
    
    PreparedStatement		getStichwortColor;
    PreparedStatement		getStichwortColorArray;

    public static final String     FILTER_STRING="&";

    /** Creates a new instance of DBAbfrage */
    public DBSTWListe(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
        dlgItem = new DialogItem(0,"",0.0,"","","","","","",null);
        HGColors = getHGArray();
    }

    private void setStatements(){
        try{
            getStichwort        = con.prepareStatement("SELECT * FROM stichworte WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getStichwortListe   = con.prepareStatement("SELECT * FROM stichworte ORDER BY stichwort",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getStichwortListeFilter   = con.prepareStatement("SELECT * FROM stichworte WHERE stichwort REGEXP ? ORDER BY stichwort",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveStichwort       = con.prepareStatement("INSERT INTO stichworte (stichwort,hg,vg) values (?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteStichwort     = con.prepareStatement("DELETE FROM stichworte WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getStichwortUsed    = con.prepareStatement("SELECT stichworte FROM rezeptliste WHERE stichworte like ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            checkStichwort      = con.prepareStatement("SELECT stichwort FROM stichworte WHERE stichwort=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            updateStichwort   = con.prepareStatement("UPDATE stichworte set stichwort=?,hg=?,vg=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

            getClientfilter = con.prepareStatement("SELECt * FROM " + DBFactory.TABLE_CLIENTFILTER,ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            setClientfilter = con.prepareStatement("UPDATE " + DBFactory.TABLE_CLIENTFILTER + " SET stichworte=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

            getStichwortColor    = con.prepareStatement("SELECT hg FROM stichworte WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getStichwortColorArray    = con.prepareStatement("SELECT id, hg FROM stichworte order by id",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            updateAblageFilter = con.prepareStatement("UPDATE " + DBFactory.TABLE_ABLAGEFILTER + " SET stichworte=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getAblageFilter = con.prepareStatement("SELECt stichworte FROM " + DBFactory.TABLE_ABLAGEFILTER,ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            adressenStatement = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWLISTE.setStatements", e.getLocalizedMessage());
        }
    }


     public Hashtable getHGArray() {
       final Hashtable hs = new Hashtable();
        try {
         final ResultSet rs = getStichwortColorArray.executeQuery();
         while(rs.next()) {
        	 	hs.put(rs.getInt("id"),rs.getString("hg"));
         }
     } catch(final Exception e) {
         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbGetStichwortColors", e.getLocalizedMessage());
     }
    	return hs;
    }
    /**
     * holt die Hintergrundfarbe aus einem Stichwort  und gibt sie als HTML-String zurück
     */

    public String getStichwortColor(int id) {
        final String ret = (String)HGColors.get(id);
        if(!ret.equals("ffffff")) {
        		return  "&nbsp;<span style=\"background-color:#" + ret +"; \">"  + "&nbsp;&nbsp;&nbsp;"+ "</span>";
        }  else {
        		return "";
        }
    }


    /**
     * holt die Filterliste für Zugangs-client-programme
     */
    public StichwortListe getZugangsClientFilter() {
        StichwortListe sl = new StichwortListe();
        try {
            final ResultSet rs = getClientfilter.executeQuery();
            if(rs.first()) {
                String zugang= rs.getString("stichworte");
                if((zugang.length() > 0) && (zugang.indexOf(FILTER_STRING) != -1)) {
                    zugang = zugang.substring(zugang.indexOf(FILTER_STRING)+1);
                } else {
                    zugang = "";
                }
                sl = dbGetStichwortListe(zugang);
            }
            if(rs != null) {
                rs.close();
            }
        } catch(final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbGetZugangsClientListe", e.getLocalizedMessage());
        }
        return sl;
    }


    /**
     * holt die Filterliste für Zugangs-client-programme
     */
    public StichwortListe getAblageFilter() {
        StichwortListe sl = new StichwortListe();
        try {
            final ResultSet rs = getAblageFilter.executeQuery();
            if(rs.first()) {
                String ablage= rs.getString("stichworte");
                sl = dbGetStichwortListe(ablage);
            }
            if(rs != null) {
                rs.close();
            }
        } catch(final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbGetZugangsClientListe", e.getLocalizedMessage());
        }
        return sl;
    }
    
    /**
     * holt die Filterliste für Abgangs-client-programme
     */
    public StichwortListe getAbgangsClientFilter() {
        StichwortListe sl = new StichwortListe();
        try {
            final ResultSet rs = getClientfilter.executeQuery();
            if(rs.first()) {
                String abgang= rs.getString("stichworte");
                if((abgang.length() > 0) && (abgang.indexOf(FILTER_STRING) != -1)) {
                    abgang = abgang.substring(0,abgang.indexOf(FILTER_STRING));
                }
                sl = dbGetStichwortListe(abgang);
            }
            if(rs != null) {
                rs.close();
            }
        } catch(final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbGetAbgangsClientListe", e.getLocalizedMessage());
        }
        return sl;
    }

    /**
     * speichert die Client-Filterliste
     */
    public void saveClientFilter(StichwortListe abgang,StichwortListe zugang) {
        try {
           String filter = abgang.toString();
           if((zugang.toString() != null) && !zugang.toString().equals("")){
               filter += FILTER_STRING + zugang.toString();
           }
           setClientfilter.setString(1,filter);
           setClientfilter.executeUpdate();
        } catch(final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbSaveClientListe", e.getLocalizedMessage());
        }
    }

    public void saveAblageFilter(StichwortListe ablage) {
        try {
           String filter = ablage.toString();
           updateAblageFilter.setString(1,filter);
           updateAblageFilter.executeUpdate();
        } catch(final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbSaveClientListe", e.getLocalizedMessage());
        }
    }
    
    /**--------------------------------------------------------------------
     * dbgetStichwortUsed() pr�ft, ob ein Stichwort in einer Rezeptur oder
     * einer Notiz verwendet wird
     * gibt true zur�ck, wenn verwendet
     *---------------------------------------------------------------------
     */
    public boolean dbGetStichwortUsed(StichwortItem si){
        boolean ret = false;
        ResultSet rs;
        final String  check = "%<" + String.valueOf(si.getID()) + ">%";
        try{
            getStichwortUsed.setString(1, check);
            getStichwortUsed.executeQuery();
            rs = getStichwortUsed.getResultSet();
            ret = rs.first();
            if(rs != null) {
				rs.close();
			}
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbGetStichwortUsed", e.getLocalizedMessage());
        }
        return ret;
    }

    /**--------------------------------------------------------------------
     * dbDeleteStichwort() löscht ein Stichwort aus der Datenbank
     * zuerst wird mit dbGetStichwortUsed überprüft, ob das Stichwort
     * verwendet wird. Wenn ja, wird das Stichwort nicht gelöscht !
     *---------------------------------------------------------------------
     */
    public void dbDeleteStichwort(StichwortItem si){
        if (!dbGetStichwortUsed(si)){
            try{
                deleteStichwort.setInt(1,si.getID());
                deleteStichwort.executeUpdate();
            }
            catch (final Exception e){
                rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.dbDeleteStichwort", e.getLocalizedMessage());
            }
        }
        else{
            dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
            dlgItem.setCaption(rz.getLocale().getString("database.stichwort_loeschen"));
            dlgItem.setTitel(rz.getLocale().getString("database.stichwort_loeschen"));
            dlgItem.setMessage(rz.getLocale().getString("database.error_stichwort_existiert").replaceAll("%s",si.getName()));
            dlgItem.setIcon("warnung.png");
            dlgItem.setOk(rz.getLocale().getString("string_ok"));
            rz.getDialogFactory().getDialog(dlgItem);
        }
        HGColors = getHGArray();
    }

    /**--------------------------------------------------------------------
     * dbCheckStichwort(String) pr�ft die Existenz eines Stichworts
     * gibt true zur�ck, wenn das Stichwort in der Datenbank existiert
     *---------------------------------------------------------------------
     */
    public boolean dbCheckStichwort(String stw){
        boolean ret = false;
        ResultSet rs;
        try{
            checkStichwort.setString(1, stw);
            checkStichwort.executeQuery();
            rs = checkStichwort.getResultSet();
            ret = rs.first();
            if(rs != null) {
				rs.close();
			}
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbGetStichwort", e.getLocalizedMessage());
        }
        return ret;
    }

    /**
     * Ein Stichwort umbenennen
     * @param dbKEy  key des Stichworts
     * @param neuer Inhalt
     * @return nummer des geänderteten Datensatzes
     */
    public int dbUpdateStichwort(StichwortItem si){
        int ret = 0;
        if((si != null) && (si.getID() != 0)) {
	            try{
	                updateStichwort.setString(1, si.getName().trim());
	                updateStichwort.setString(2,si.getHg().trim());
	                updateStichwort.setString(3,si.getVg().trim());
	                updateStichwort.setInt(4,si.getID());
	                updateStichwort.executeUpdate();
	                ret = db.dbGetLastInsertID("stichworte");
	            }
	            catch (final Exception e){
	                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbRenameStichwort", e.getLocalizedMessage());
	            }
	        }
	        else{
	            dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
	            dlgItem.setCaption(rz.getLocale().getString("stichwort.speichern"));
	            dlgItem.setTitel(rz.getLocale().getString("stichwort.speichern"));
	            dlgItem.setMessage(rz.getLocale().getString("database.error_stichwort_nosave").replaceAll("%s",si.getName()));
	            dlgItem.setIcon("warnung.png");
	            dlgItem.setOk(rz.getLocale().getString("string_ok"));
	            rz.getDialogFactory().getDialog(dlgItem);
	        }
        HGColors = getHGArray();
        return ret;
    }

    /**
     * dbSaveStichwort(Stichwortitem) fügt ein Stichwort in die Datenbank ein
     * zuerst wird mit dbCheckStichwort überprüft, ob das Stichwort
     * verwendet wird. Wenn ja, wird kein neues Stichwort hinzugefügt
     *
     */
    public int dbSaveStichwort(StichwortItem si){
        int ret = 0;
        if (!dbCheckStichwort(si.getName().trim())){
            try{
	                saveStichwort.setString(1,  si.getName().trim());
	                saveStichwort.setString(2,si.getHg());
	                saveStichwort.setString(3,si.getVg());
	                saveStichwort.executeUpdate();
	                ret = db.dbGetLastInsertID("stichworte");
            }
	         catch (final Exception e){
	             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBSTWListe.dbSaveStichwort", e.getLocalizedMessage());
            }
        }
        else{
            dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
            dlgItem.setCaption(rz.getLocale().getString("stichwort.speichern"));
            dlgItem.setTitel(rz.getLocale().getString("stichwort.speichern"));
            dlgItem.setMessage(rz.getLocale().getString("database.error_stichwort_nosave").replaceAll("%s",si.getName()));
            dlgItem.setIcon("warnung.png");
            dlgItem.setOk(rz.getLocale().getString("string_ok"));
            rz.getDialogFactory().getDialog(dlgItem);
        }
        HGColors = getHGArray();
        return ret;
    }

    /**
     * dbGetStichwortListe()
     * holt die gesamte Stichwortliste aus der Datenbank
     *
     */
     public StichwortListe dbGetStichwortListe(){
         final StichwortListe sl = new StichwortListe();
         ResultSet rs;
         try{
             getStichwortListe.executeQuery();
             rs = getStichwortListe.getResultSet();
             while(rs.next()){
                 sl.addItem(new StichwortItem(rs.getInt("id"), rs.getString("stichwort"),rs.getString("hg"),rs.getString("vg")));
             }
             if(rs != null) {
				rs.close();
			}
         }
         catch (final Exception e){
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.getStichwortListe", e.getLocalizedMessage());
         }
         return sl;
     }

     public StichwortListe dbGetStichwortListeFilter(String filter){
         final StichwortListe sl = new StichwortListe();
         ResultSet rs;
         try{
        	 if (!filter.trim().equals("^")) {
        		 filter = filter.replaceAll("\\s",".*");
        		 getStichwortListeFilter.setString(1,filter);
             	getStichwortListeFilter.executeQuery();
             	rs = getStichwortListeFilter.getResultSet();
             	while(rs.next()){
            	 	sl.addItem(new StichwortItem(rs.getInt("id"), rs.getString("stichwort"),rs.getString("hg"),rs.getString("vg")));
             	}
             	if(rs != null) {
					rs.close();
				}
             }
         }
         catch (final Exception e){
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.getStichwortListe", e.getLocalizedMessage());
         }
         return sl;
     }

     public StichwortItem dbGetStichwort(int id) {
        StichwortItem si = new StichwortItem();
        try {
        	if(id != 0) {
        		ResultSet rs;
                getStichwort.setInt(1,id);
                getStichwort.executeQuery();
                rs = getStichwort.getResultSet();
                if (rs.next()){
                	si=new StichwortItem(rs.getInt("id"),rs.getString("stichwort"),rs.getString("hg"),rs.getString("vg"));
                }
                if(rs != null) {
					rs.close();
				}
        	}
        }
        catch (final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.dbGetStichwort", e.getLocalizedMessage());
        }
        return si;
    }
    /**
      * dbGetStichwortListe(String)
      * holt eine Stichwortliste aus einem
      * StichwortString z.B. "<1><4>"
      *
      */
      public StichwortListe dbGetStichwortListe(String s){
          final StichwortListe al = new StichwortListe();
          ResultSet rs;
          s = s.replaceAll("[<|>]"," ").replaceAll("%","");
          try {
              final StringTokenizer stt = new StringTokenizer(s);
              while (stt.hasMoreTokens()){
                  final int k = Integer.parseInt( stt.nextToken());
                  getStichwort.setInt(1,k);
                  getStichwort.executeQuery();
                  rs = getStichwort.getResultSet();
                  if (rs.first()){
                      al.addItem(new StichwortItem(k,rs.getString("stichwort"),rs.getString("hg"),rs.getString("vg")));
                  }
                  if(rs != null) {
					rs.close();
				}
              }
          }
          catch (final SQLException e){
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.getStichwortListe:String:", e.getLocalizedMessage());
          }
          al.sortItems();
          return al;
      }


      /**
       * dbGetAdressenListe(String)
       * holt eine Adressenliste aus einem
       * StichwortString z.B. "<1><4>"
       *
       */
       public StichwortListe dbGetAdressenListe(String s){
           final StichwortListe al = new StichwortListe();
           ResultSet rs=null;
           s = s.replaceAll("[<|>]"," ").replaceAll("%","");
           try {
        	   String liste="";
               final StringTokenizer stt = new StringTokenizer(s);
               while (stt.hasMoreTokens()){
                   final int k = Integer.parseInt( stt.nextToken());
                   liste += String.valueOf(k)+",";
               	}
               if(liste.endsWith(","))liste = liste.substring(0,liste.length()-1);
               if(!liste.equals("")){
            	   liste = "SELECT * FROM adressen WHERE id in(" + liste + ") ORDER BY id";
            	   
                   rs = adressenStatement.executeQuery(liste);
                   while(rs.next()){
                       al.addItem(new StichwortItem(rs.getInt("id"),rs.getString("firma"),"",""));
                   }
               }
               if(rs != null) {
 					rs.close();
 				}
           }
           catch (final SQLException e){
              rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.getAdressenListe", e.getLocalizedMessage());
           }
           return al;
       }
      
      public String getHTMLBarString(String s) {
    	 String ret ="";
          s = s.replaceAll("[<|>]"," ").replaceAll("%","");
          try {
              final StringTokenizer stt = new StringTokenizer(s);
              while (stt.hasMoreTokens()){
                  ret += getStichwortColor(Integer.parseInt( stt.nextToken()));
              }
          }
          catch (final Exception e){
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"DBSTWListe.getHTMLBarString", e.getLocalizedMessage());
          }
    	  return ret;
      }
}
