/*
 *  Copyright (C) database.stichwort_loeschen4 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.rmi.server.UID;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.items.lieferschein.AuftragsListe;
import de.lunqual.rzpro.items.lieferschein.LieferscheinEinfachItem;
import de.lunqual.rzpro.items.lieferschein.LieferscheinEinfachListe;
import de.lunqual.rzpro.items.lieferschein.LieferscheinItem;
import de.lunqual.rzpro.items.lieferschein.LieferscheinZeile;
import de.lunqual.rzpro.log.LogFactory;
/**
 *
 * @author  lunqual
 */
public class DBLieferschein {

    RzPro           rz;
    Connection      con;
    DBFactory       db;

    PreparedStatement   	getLieferschein;
    PreparedStatement		getLieferscheinListe;
    PreparedStatement		getLieferscheinListeOhne;
    PreparedStatement	 	getMaxID;
    PreparedStatement		saveLieferschein;
    PreparedStatement		updateLieferschein;
    PreparedStatement		saveLieferscheinZeile;
    PreparedStatement 		getLieferscheinComplete;
    PreparedStatement		getContainerListe;
    PreparedStatement		updateLieferscheinZeile;
    PreparedStatement		deleteLieferscheinZeile;
    PreparedStatement		deleteLieferschein;
    PreparedStatement		getLieferscheinLosnummer;
    PreparedStatement		getLieferscheinNummer;
    PreparedStatement		updateZeilenComment;
    PreparedStatement		getEtikettenChars;
    PreparedStatement		setEtikettenChars;
    PreparedStatement		getLieferscheinListeAdresseProdukt;
    
    public static final int	ETIKETTEN_MUSTER_REMOVE=0;
    public static final int	ETIKETTEN_MUSTER_MUSTER=1;
    
    /** Creates a new instance of DBAbfrage */
    public DBLieferschein(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
    }

    private void setStatements(){
        try{
            getLieferschein  = con.prepareStatement("SELECT *,lieferschein.comment as comment,lieferschein_zeile.comment as zcomment FROM " + DBFactory.TABLE_LIEFERSCHEIN + " RIGHT JOIN lieferschein_zeile ON lieferschein_zeile.lieferschein = lieferschein.id WHERE lieferschein.id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLieferscheinListe  = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_LIEFERSCHEIN + " WHERE adresse=? order by  geaendert DESC",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLieferscheinListeOhne  = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_LIEFERSCHEIN + " WHERE  adresse_string REGEXP ? order by geaendert DESC ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getMaxID			= con.prepareStatement("SELECT MAX( id ) as max FROM " + DBFactory.TABLE_LIEFERSCHEIN ,ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveLieferschein = con.prepareStatement("INSERT INTO " + DBFactory.TABLE_LIEFERSCHEIN  + " (nummer,adresse,adresse_string,erstellt,geaendert,user_1,user_2,comment) VALUES (?,?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateLieferschein = con.prepareStatement("UPDATE " + DBFactory.TABLE_LIEFERSCHEIN  + " set nummer=?,adresse=?,adresse_string=?,geaendert=?,user_2=?,comment=?  WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            saveLieferscheinZeile = con.prepareStatement("INSERT INTO " + DBFactory.TABLE_LIEFERSCHEIN_ZEILE + " (lieferschein,bezeichnung,rezeptur,containerS,container,losnummer,staerke,litergewicht,brutto,tara,netto,liter,kg,la,comment,ean) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getLieferscheinComplete  = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_LIEFERSCHEIN + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getContainerListe = con.prepareStatement("SELECT id,lieferschein FROM " + DBFactory.TABLE_LIEFERSCHEIN_ZEILE + " where lieferschein=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            deleteLieferscheinZeile = con.prepareStatement("delete from " + DBFactory.TABLE_LIEFERSCHEIN_ZEILE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateLieferscheinZeile = con.prepareStatement("UPDATE " + DBFactory.TABLE_LIEFERSCHEIN_ZEILE + " set lieferschein=?,bezeichnung=?,rezeptur=?,containerS=?,container=?,losnummer=?,staerke=?,litergewicht=?,brutto=?,tara=?,netto=?,liter=?,kg=?,la=?,comment=?,ean=?  WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteLieferschein = con.prepareStatement("DELETE FROM " + DBFactory.TABLE_LIEFERSCHEIN + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteLieferscheinZeile = con.prepareStatement("DELETE FROM " + DBFactory.TABLE_LIEFERSCHEIN_ZEILE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getLieferscheinLosnummer  = con.prepareStatement( "select distinct lieferschein.id,lieferschein.nummer,lieferschein.adresse_string from lieferschein left join lieferschein_zeile on lieferschein_zeile.lieferschein = lieferschein.id where lieferschein_zeile.losnummer regexp ? order by lieferschein.geaendert desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLieferscheinNummer  = con.prepareStatement( "select distinct lieferschein.id,lieferschein.nummer,lieferschein.adresse_string from lieferschein left join lieferschein_zeile on lieferschein_zeile.lieferschein = lieferschein.id where lieferschein.nummer regexp ? order by lieferschein.geaendert desc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLieferscheinListeAdresseProdukt  = con.prepareStatement( "select distinct lieferschein.id,lieferschein.nummer,lieferschein.adresse_string,lieferschein_zeile.bezeichnung from lieferschein left join lieferschein_zeile on lieferschein_zeile.lieferschein = lieferschein.id where lieferschein.adresse_string regexp ? AND lieferschein_zeile.bezeichnung regexp ? order by lieferschein.geaendert desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
          
            getEtikettenChars  = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_LIEFERSCHEIN_L_MUSTER +" WHERE id=1 " ,ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            setEtikettenChars = con.prepareStatement("UPDATE  " + DBFactory.TABLE_LIEFERSCHEIN_L_MUSTER + " set remove=?,muster=? WHERE id=1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".setStatements", e.getLocalizedMessage());
        }
    }

    public String getLosnummerVorschlag(String remove,String muster,String losnummer,Date datum) {
        String ret = "";
        String template;
        if(losnummer.contains(remove)) {
        	template = muster;
        }else {
        	template = losnummer;
        }
        Calendar c = Calendar.getInstance();
        c.setTime(datum);
        if(!remove.equals("")) {
        	losnummer=losnummer.replaceAll(remove, "");
        }
        final SimpleDateFormat monat = new SimpleDateFormat("MM");
        final SimpleDateFormat jahr2 = new SimpleDateFormat("yy");
        final SimpleDateFormat jahr4 = new SimpleDateFormat("yyyy");
        final SimpleDateFormat julian = new SimpleDateFormat("D");
        /* templates ersetzen
         * $w = Tag des Jahres dezimal
         * $W = Tag des Jahres duodezimal
         * $m = Monat
         * $y = Jahr (2-stellig)
         * $Y = Jahr (4-stellig)
         * $J = Jahr duodezimal
         * $u = Benutzer (vollständig)
         * $U = Benutzer (nur 1 Buchstabe)
         * $l = originalLosnummer
        */
        String ab ="";
        String ab4 = "";

        ret = template
           	.replaceAll("\\$J",rz.getDatabase().getKalender().getYear20(c))
            .replaceAll("\\$w",julian.format(datum))
            .replaceAll("\\$W",rz.getDatabase().getKalender().getDate20(c))
        	.replaceAll("\\$m",monat.format(datum))
        	.replaceAll("\\$y",jahr2.format(datum))
        	.replaceAll("\\$Y",jahr4.format(datum))
        	.replaceAll("\\$l",losnummer.trim())
        	.replaceAll("\\$b",ab)
        	.replaceAll("\\$4",ab4)
        	.replaceAll("\\$u",db.dbGetUser())
        	.replaceAll("\\$U",String.valueOf(db.dbGetUser().charAt(0))

        			
        			);
        // jetzt noch alle übriggebliebenenen Verbotenen Zeichen entfernen
        //ret = rz.getOptionFactory().deleteInvalidChars(ret);
        return ret;
    }
    
    
    public String getEtikettenChars(int what) {
    	String ret="";
    	try {
    		ResultSet rs=getEtikettenChars.executeQuery();
    		if(rs.next()) {
    			switch(what) {
    				case ETIKETTEN_MUSTER_REMOVE: ret=rs.getString("remove"); break;
    				case ETIKETTEN_MUSTER_MUSTER: ret=rs.getString("muster"); break;
    			}
    		}
    	}catch(Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN_L_MUSTER + ".lieferschein.getEtikettenChars", e.getLocalizedMessage());
    	}
    	
    	return ret;
    }
    
    public void setEtikettenChars(String remove,String muster) {
    	try {
    		setEtikettenChars.setString(1,remove);
    		setEtikettenChars.setString(2, muster);
    		setEtikettenChars.executeUpdate();
    	}catch(Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN_L_MUSTER + ".lieferschein.getEtikettenChars", e.getLocalizedMessage());
    	}
    }
    
    /** speichert ein LieferscheinItem oder macht ein Update für eine bestehendes LieferscheinItem
     *
     * @param li   LieferscheinItem  (wenn id==0 speichern, id != 0 update)
     * @return true, wenn gespeichert werden konnte
     */
    public boolean saveLieferschein(LieferscheinItem li) {
        if(li.getId() == 0) {
            return lieferscheinSave(li);
        }
		return lieferscheinUpdate(li);
    }


    
    /** speichert ein LieferscheinItem in der Datenbank ab
     * Dabei wird die Containerliste in einer Transaktion mit gespeichert
     *
     * @param li LieferscheinItem
     * @return true, wenn alles gespeichert werden konnte.
     */
    private boolean lieferscheinSave(LieferscheinItem li) {
        boolean ret = true;
        try {
            con.setAutoCommit(false);
            saveLieferschein.setString(1,li.getNummer());
            saveLieferschein.setInt(2,li.getAdresse());
            saveLieferschein.setString(3,li.getAdresse_string());
            saveLieferschein.setString(4,db.dbGetDateFormatString(li.getErstellt()));
            saveLieferschein.setString(5,db.dbGetDateFormatString(li.getGeaendert()));
            saveLieferschein.setString(6,db.dbGetUser());
            saveLieferschein.setString(7, db.dbGetUser());
            saveLieferschein.setString(8,li.getComment());
            saveLieferschein.executeUpdate();
            li.setId(db.dbGetLastInsertID(DBFactory.TABLE_LIEFERSCHEIN));
            ret = containerSave(li);
            if(!ret) {
                con.rollback();
            } else {
            	con.commit();
            }
            con.setAutoCommit(true);
        } catch(final Exception ex) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferscheinSave.rollback", ex.getLocalizedMessage());
            ret = false;
            if (con != null) {
	            try {
		            con.rollback();
		            con.setAutoCommit(true);
		            } catch(final Exception excep) {
		                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferscheinSave.2", ex.getLocalizedMessage());
		                ret = false;
		            }
	            }
            }
        return ret;
    }

    /** speichert einen einen einzelnen Container (in der LieferscheinZeile enthalten)
     * in der Datenbank ab.
     *
     * @param lieferschein  der Primärschlüssel des zugehörigen Lieferscheins
     * @param zl  LieferscheinZeile (einzelner Container)
     * @return
     */
    public boolean containerSave(int lieferschein,LieferscheinZeile zl) {
        boolean ret = true;
        try {
                saveLieferscheinZeile.setInt(1,lieferschein);
                saveLieferscheinZeile.setString(2,zl.getBezeichnung());
                saveLieferscheinZeile.setInt(3, zl.getRezeptur().getID());
                saveLieferscheinZeile.setString(4, zl.getContainerS());
                saveLieferscheinZeile.setInt(5,zl.getContainer());
                saveLieferscheinZeile.setString(6,zl.getLosnummer());
                saveLieferscheinZeile.setDouble(7, zl.getStaerke());
                saveLieferscheinZeile.setDouble(8, zl.getLitergewicht());
                saveLieferscheinZeile.setDouble(9, zl.getBrutto());
                saveLieferscheinZeile.setDouble(10, zl.getTara());
                saveLieferscheinZeile.setDouble(11, zl.getNetto());
                saveLieferscheinZeile.setDouble(12, zl.getLiter());
                saveLieferscheinZeile.setDouble(13, zl.getLiter());
                saveLieferscheinZeile.setDouble(14, zl.getLa());
                saveLieferscheinZeile.setString(15,zl.getComment());
                saveLieferscheinZeile.setString(16, zl.getEan());
            	saveLieferscheinZeile.executeUpdate();
            	final int id =db.dbGetLastInsertID(DBFactory.TABLE_LIEFERSCHEIN_ZEILE);
            	zl.setId(id);
            	if(zl.getContainer() != 0) {
            		rz.getDatabase().getContainer().setStatus(zl.getContainer(),1);
            	}
        }catch (final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferschein.containerSave:Zeile", e.getLocalizedMessage());
            ret = false;
        }
        return ret;
    }

    private boolean containerSave(LieferscheinItem li) {
        boolean ret = true;
        try {
            final ArrayList al = li.getAuftragsListe().getContainerListe(AuftragsListe.MODE_ALLE);
            for(int i = 0;i < al.size();i++) {
                final LieferscheinZeile zl = (LieferscheinZeile)al.get(i);
                if(!containerSave(li.getId(),zl)) {
                    ret = false;
                    break;
                }
            }
        }catch (final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferschein.containerSave:item", e.getLocalizedMessage());
            ret = false;
        }
        return ret;
    }

    /**
     * die Containerliste eines Lieferscheins updaten.
     * Um es komplizierter zu machen, wird der Platz in der Datenbank wo möglich
     * recycled
     * Im ersten Durchgang wird die Liste der Container aus der Datenbank gezogen,
     * die VORHER zum Lieferschein gehört haben. Es wird geprüft ob Einträge dabei sind, die
     * JETZT nicht mehr zum Lieferschein gehören... Diese Einträge werden dann aus der
     * Datenbank gelöscht.
     *
     * Dann werden alle LieferscheinContainer mit der ID != 0 upgedatet.
     *
     * Zuletzt werden Container (mit id == 0) neu in der Datenbank angelegt.
     *
     * @param li		LieferscheinItem
     * @return true, wenn keine Fehler aufgetreten sind
     */
    private boolean containerUpdate(LieferscheinItem li) {
        boolean ret = true;
        int i;
       final ArrayList vorhanden = new ArrayList();
        try {
            ResultSet rs;
            /* zuerst stellen wir mal fest, welche Container bisher zu dem Lieferschein gehört haben, so dass
             * wir nicht mehr vorhandene Zeilen löschen können
             */
            getContainerListe.setInt(1,li.getId());
            rs = getContainerListe.executeQuery();
            while (rs.next()) {
                vorhanden.add(Integer.valueOf(rs.getInt("id")));
            }
            /* jetzt arbeiten wir die Liste ab. vorhandene IDs werden upgedated, was übrigbleibt
             * wird aus der Datenbank entfernt
             */
            for( i = 0;i < vorhanden.size();i++) {
                final int index = ((Integer)vorhanden.get(i)).intValue();
                if (index != 0) { // just to be sure, dürfte eigentlich nicht passieren, aber mein Arzt sagt........
                    final LieferscheinZeile zl = li.getAuftragsListe().getContainer(index);
                    if(zl != null) {
                        updateLieferscheinZeile.setInt(1,li.getId());
                        updateLieferscheinZeile.setString(2,zl.getBezeichnung());
                        updateLieferscheinZeile.setInt(3,zl.getRezeptur().getID());
                        updateLieferscheinZeile.setString(4, zl.getContainerS());
                        updateLieferscheinZeile.setInt(5,zl.getContainer());
                        updateLieferscheinZeile.setString(6,zl.getLosnummer());
                        updateLieferscheinZeile.setDouble(7, zl.getStaerke());
                        updateLieferscheinZeile.setDouble(8,zl.getLitergewicht());
                        updateLieferscheinZeile.setDouble(9, zl.getBrutto());
                        updateLieferscheinZeile.setDouble(10, zl.getTara());
                        updateLieferscheinZeile.setDouble(11, zl.getNetto());
                        updateLieferscheinZeile.setDouble(12, zl.getLiter());
                        updateLieferscheinZeile.setDouble(13, zl.getLiter());
                        updateLieferscheinZeile.setDouble(14, zl.getLa());
                        updateLieferscheinZeile.setString(15,zl.getComment());
                        updateLieferscheinZeile.setString(16, zl.getEan());
                        updateLieferscheinZeile.setInt(17,zl.getId());
                    	updateLieferscheinZeile.executeUpdate();
                    	if(zl.getContainer() != 0) {
                    		rz.getDatabase().getContainer().setStatus(zl.getContainer(),1);
                    	}
                    }else {
                        // löschen, die ID ist nicht mehr auf dem Lieferschein !
                        deleteLieferscheinZeile.setInt(1,index);
                        deleteLieferscheinZeile.executeUpdate();
                    }
                }
            }
            /** und jetzt den Lieferschein nochmal durchgehen, um evtl neue Container
             * noch zu speichern
             */
            final ArrayList al = li.getAuftragsListe().getContainerListe(AuftragsListe.MODE_NEUE);
            for(i = 0;i < al.size();i++) {
                if(containerSave(li.getId(),(LieferscheinZeile)al.get(i)) == false) {
                	ret = false;
                	break;
                }
            }
            if(rs != null) {
				rs.close();
			}
        } catch (final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferschein.containerUpdate.", e.getLocalizedMessage());
            ret = false;
        }
        return ret;
    }

    /**
     * LieferscheiItem updaten
     * @param li Lieferscheinitem
     * @return  true, wenn keine Fehler aufgetreten sind
     */
    private boolean lieferscheinUpdate(LieferscheinItem li) {
        boolean ret = true;
        try {
            con.setAutoCommit(false);
            updateLieferschein.setString(1,li.getNummer());
            updateLieferschein.setInt(2,li.getAdresse());
            updateLieferschein.setString(3, li.getAdresse_string());
            updateLieferschein.setString(4,db.dbGetDateFormatString(db.getServerTimestamp()));
            updateLieferschein.setString(5, db.dbGetUser());
            updateLieferschein.setString(6,li.getComment());
            updateLieferschein.setInt(7,li.getId());
            updateLieferschein.executeUpdate();
            ret = containerUpdate(li);
            if(ret) {
                con.commit();
            } else {
            	con.rollback();
            }
            con.setAutoCommit(true);
        } catch(final Exception ex) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferscheinUpdate.rollback", ex.getLocalizedMessage());
            ret = false;
            if (con != null) {
	            try {
		            con.rollback();
		            con.setAutoCommit(true);
		            } catch(final Exception excep) {
		                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferscheinUpdate.2", ex.getLocalizedMessage());
		                ret = false;
		            }
	            }
            }
        return ret;
    }

    /**
     * einen Lieferschein aus der Datenbank holen
     * @param id  Primärschlüssel des Lieferscheins
     * @param quiet  gibt an, ob nicht mehr existente Einträge angemahnt werden
     * @return LieferscheinItem
     */
    public LieferscheinItem getLieferschein(int id,boolean quiet) {
        LieferscheinItem li = null;
        try {
            if(id == 0) {
                li = new LieferscheinItem(
                        rz,
                        0,
                        getNextLieferscheinNummer(),
                        0,
                        "",
                        db.getServerTimestamp(),
                        db.getServerTimestamp(),
                        db.dbGetUser(),
                        db.dbGetUser(),
                        ""
                );
            } else {
                if(db.idExists(DBFactory.TABLE_LIEFERSCHEIN, id, quiet)) {
	                getLieferschein.setInt(1,id);
	                final ResultSet rs = getLieferschein.executeQuery();
	                if(rs.next()) {
	                    li = new LieferscheinItem(
	                            rz,
	                            rs.getInt("id"),
	                            rs.getString("nummer"),
	                            rs.getInt("adresse"),
	                            rs.getString("adresse_string"),
	                            db.getDateFromString(rs.getString("erstellt")),
	                            db.getDateFromString(rs.getString("geaendert")),
	                            rs.getString("user_1"),
	                            rs.getString("user_2"),
	                            rs.getString("comment")
	                    );
	                    li.getAuftragsListe().addContainer(getLieferscheinZeile(rs,li));
	                    while(rs.next()) {
		                    li.getAuftragsListe().addContainer(getLieferscheinZeile(rs,li));
	                    }
	                }
	                if(rs != null) {
						rs.close();
					}
                }
            }
        } catch (final Exception se) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + "getLieferschein", se.getLocalizedMessage());
        }
        return li;
    }

    LieferscheinZeile getLieferscheinZeile(ResultSet rs,LieferscheinItem li) {
        try {
	        return new LieferscheinZeile(
	                rz,
	                new UID(),
	                li.getId(),
	                rs.getInt("lieferschein_zeile.id"),
	                rs.getString("bezeichnung"),
	                db.getRezeptur().dbGetRezeptur(rs.getInt("rezeptur"),true),
	                rs.getInt("container"),
	                rs.getString("containerS"),
	                DBBuchung.REIHE_FIFO,
	                rs.getString("losnummer"),
	                rs.getDouble("staerke"),
	                rs.getDouble("litergewicht"),
	                rs.getDouble("brutto"),
	                rs.getDouble("tara"),
	                rs.getDouble("netto"),
	                rs.getDouble("liter"),
	                rs.getDouble("kg"),
	                rs.getDouble("la"),
	                rs.getString("zcomment"),
	                rs.getString("ean"),
	                li.getErstellt()
	        );
        } catch (final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + "getLieferscheinZeile", e.getLocalizedMessage());
            return null;
        }
    }
    /**
     * holt eine liste aus einfachen LieferscheinItems nach Auswahl einer Firma
     * @param filter   String, nach dem die Adressen festgelegt werden
     * @param mitAdressen, wenn true, wird die Liste aus Lieferscheinen geliefert,
     * die Adressen zugeordnet sind, wenn false, wird die Liste der Lieferscheine ohne Adressen
     * zurückgeliefert.
     *
     * @return
     */
    public LieferscheinEinfachListe getEinfachListe(String filter) {
        boolean produkt = false;
        String sorte = "";
        final LieferscheinEinfachListe ll = new LieferscheinEinfachListe();
        ResultSet rs = null;
        try {
    	   if(filter.contains("?")) {
    		   String items[] = filter.split("\\?");
    		   filter = items[0].trim();
    		   sorte = items[1].trim();
    		   if(!sorte.equals("")) {
    			   produkt=true;
    		   }
    	   }
    	   filter = filter.replaceAll("\\s+",".*");
    	   sorte= sorte.replaceAll("\\s+",".*");
           if(filter.equals("")) {
				filter = ".*";
			}
    	   if(produkt) {
               getLieferscheinListeAdresseProdukt.setString(1,filter);
               getLieferscheinListeAdresseProdukt.setString(2,sorte);
        	   rs = getLieferscheinListeAdresseProdukt.executeQuery();
           }
           else {
               getLieferscheinListeOhne.setString(1,filter);
        	   rs = getLieferscheinListeOhne.executeQuery();
           }
           while(rs.next()) {
                ll.addItem(new LieferscheinEinfachItem(
                rs.getInt("id"),
                rs.getString("nummer"),
                0,
                rs.getString("adresse_string")
               	));
           }
           if(rs != null) {
			rs.close();
		}

        } catch (final Exception se) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".getLieferscheinListeEinfach", se.getLocalizedMessage());
        }
        return ll;
    }


    public LieferscheinEinfachListe getLosnummerListe(String losnummer) {
    	LieferscheinEinfachListe ll=new LieferscheinEinfachListe();
    	try {
    		if(!losnummer.trim().equals("")) {
    			getLieferscheinLosnummer.setString(1,losnummer);
    			ResultSet rs = getLieferscheinLosnummer.executeQuery();
    			while(rs.next()) {
    				  ll.addItem(new LieferscheinEinfachItem(
    			                rs.getInt("id"),
    			                rs.getString("nummer"),
    			                0,
    			                rs.getString("adresse_string")
    			               	));
    			}
    		}
	    } catch (final Exception se) {
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".getLieferscheinListeLosnummer", se.getLocalizedMessage());
	    }
	    return ll;
    }
    
    public LieferscheinEinfachListe getNummerListe(String nummer) {
    	LieferscheinEinfachListe ll=new LieferscheinEinfachListe();
    	try {
    		if(!nummer.trim().equals("")) {
    			getLieferscheinNummer.setString(1,nummer);
    			ResultSet rs = getLieferscheinNummer.executeQuery();
    			while(rs.next()) {
    				  ll.addItem(new LieferscheinEinfachItem(
    			                rs.getInt("id"),
    			                rs.getString("nummer"),
    			                0,
    			                rs.getString("adresse_string")
    			               	));
    			}
    		}
	    } catch (final Exception se) {
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".getLieferscheinListeLosnummer", se.getLocalizedMessage());
	    }
	    return ll;
    }
    
    /** einen Lieferschein löschen
     *
     * @param id   id des Lieferscheins
     */
    public boolean deleteLieferschein(int id) {
        final LieferscheinItem li = getLieferschein(id,false);
        if(li != null) {
            return deleteLieferschein(li);
        }
		return false;
    }

    /** die Containerliste des Lieferscheins löschen
     *
     * @param li
     * @return
     */
    private boolean deleteContainerListe(LieferscheinItem li) {
        boolean ret = true;
        try {
           getContainerListe.setInt(1,li.getId());
           final ResultSet rs = getContainerListe.executeQuery();
           while (rs.next()) {
               deleteLieferscheinZeile.setInt(1,rs.getInt("id"));
               deleteLieferscheinZeile.executeUpdate();
           }
           rs.close();
        } catch (final Exception se) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".DeleteContainerlist", se.getLocalizedMessage());
            ret = false;
        }
        return ret;
    }
    /** einen lieferschein aus der Datenbank  löschen
     *
     * @param li  LieferscheinItem
     */
    public boolean deleteLieferschein(LieferscheinItem li) {
        boolean ret = true;
        try {
            con.setAutoCommit(false);
            if(deleteContainerListe(li) == true) {
              deleteLieferschein.setInt(1,li.getId());
              if(deleteLieferschein.executeUpdate() ==0) {
                  con.rollback();
              } else {
              	con.commit();
              }
            } else {
                con.rollback();
            }
            con.setAutoCommit(true);
        } catch(final Exception ex) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferscheinDelete.rollback", ex.getLocalizedMessage());
            ret = false;
            if (con != null) {
	            try {
		            con.rollback();
		            con.setAutoCommit(true);
		            } catch(final Exception excep) {
		                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_LIEFERSCHEIN + ".lieferscheinDelete.2", ex.getLocalizedMessage());
		                ret = false;
		            }
	            }
            }
        return ret;
    }
    /** liefert die nächste freie Lieferschein-Nummer
     *
     * @return String  (<datum>-nummer)
     */
    public String getNextLieferscheinNummer() {
        String ret = "";
        final SimpleDateFormat df = new SimpleDateFormat("dd.MM.yyyy");
        try {
            final ResultSet rs = getMaxID.executeQuery();
            if(rs.next()) {
                ret += df.format(db.getServerTimestamp())  + "-" + String.valueOf(rs.getInt("max")+1);
            }
            if(rs != null) {
				rs.close();
			}
        } catch(final Exception se) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB-" + DBFactory.TABLE_LIEFERSCHEIN + "-getMaxID", se.getLocalizedMessage());
            ret += df.format(new Date())  + "-1";
        }
        return ret;
    }
}
