/*
 *  Copyright (C) database.stichwort_loeschen4 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package de.lunqual.rzpro.database;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.items.einheiten.EinheitenMap;
import de.lunqual.rzpro.items.einheiten.EinheitsItem;
import de.lunqual.rzpro.log.LogFactory;

import javax.swing.JComboBox;
/**
 * @author lunqual
 * DBEinheiten verwaltet die Einheiten Liter,Kg,LA usw...
 */
	public class DBEinheiten {
	    RzPro 	          	rz;
	    Connection      	con;
	    DBFactory       	db;
	    EinheitenMap  grundeinheiten;
	    EinheitenMap  einheiten;
	    ArrayList			einheitenListe;
	    ArrayList			grundEinheitenListe;
	    String				litergewicht;
	    String				volumen;
	    Pattern         		einheit;
	    Matcher         	einheitenMatcher;

	    Statement		getReadOnly;

	    EinheitsItem	liter;
	    EinheitsItem	kg;
	    EinheitsItem	la;

	    public DBEinheiten(RzPro r,DBFactory aDb,Connection aCon) {
	        rz = r;
	        db = aDb;
	        con = aCon;
	        setStatements();
	        openEinheiten();
	    }

	    private void setStatements(){
	        try{
	            getReadOnly     = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	        }
	        catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBEinheiten.setStatements", e.getLocalizedMessage());
	        }
	    }

	    private void openEinheiten(){
	    	ResultSet rs;
	    	grundeinheiten = new EinheitenMap(rz);
	        einheiten = new EinheitenMap(rz);
	        grundEinheitenListe = new ArrayList();
	        einheitenListe = new ArrayList();
	        try{
	          rs = getReadOnly.executeQuery("SELECT * FROM  " + DBFactory.TABLE_GRUNDEINHEITEN + "  ORDER BY id");
	          while (rs.next()){
	              grundeinheiten.addItem(
	                  new EinheitsItem(rs.getInt("id"),
	                                   rs.getString("name"),
	                                   rs.getInt("einheit"),
	                                   1));
	          }
	          rs = getReadOnly.executeQuery("SELECT * FROM " + DBFactory.TABLE_EINHEITEN + " ORDER BY id");
	          while (rs.next()){
	              einheiten.addItem(
	                  new EinheitsItem(rs.getInt("id"),
	                                   rs.getString("name"),
	                                   rs.getInt("typ"),
	                                   rs.getDouble("faktor")));
	          }
	          rs = getReadOnly.executeQuery("SELECT * FROM " + DBFactory.TABLE_EINHEITEN + " ORDER BY name");
	          while (rs.next()){
	              einheitenListe.add(
	                  new EinheitsItem(rs.getInt("id"),
	                                   rs.getString("name"),
	                                   rs.getInt("typ"),
	                                   rs.getDouble("faktor")));
	          }
	          rs = getReadOnly.executeQuery("SELECT * FROM  " + DBFactory.TABLE_GRUNDEINHEITEN + "  ORDER BY name");
	          while (rs.next()){
	              grundEinheitenListe.add(
	                  new EinheitsItem(rs.getInt("id"),
	                                   rs.getString("name"),
									   rs.getInt("id"),
									   1));
	          }
	          // Litergewichts-String erzeugen...
	          litergewicht = "";
	          volumen = "";
	          String v1 = "";String v2="";
	          rs = getReadOnly.executeQuery("SELECT * FROM  " + DBFactory.TABLE_GRUNDEINHEITEN + "  WHERE id="+RzPro.EINHEIT_KG);
	          if (rs.next()){
	              litergewicht += rs.getString("name") + "/";
	              v1 = rs.getString("name") ;
	          }
	          rs = getReadOnly.executeQuery("SELECT * FROM  " + DBFactory.TABLE_GRUNDEINHEITEN + "  WHERE id="+RzPro.EINHEIT_LITER);
	          if (rs.next()){
	              litergewicht += rs.getString("name") ;
	              v2 = rs.getString("name") ;
	          }
	          volumen = v2 + "/" + v1;
	          // Pattern zum ERkennen von gültigen Einheiten...
	          einheit = Pattern.compile("^\\s*"+getEinheitenRegex()+"\\s*$",Pattern.CASE_INSENSITIVE);

	          liter = getEinheit("liter");
	          kg = getEinheit("kg");
	          la = getEinheit("la");

	          if(rs != null) {
				rs.close();
			}
	        } catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBEinheiten.openEinheiten", e.getLocalizedMessage());
	        }
	    }

	    /** Litergewicht-einheiten Kg/Liter als String*/
	    public String dbGetLitergewichtString(){
	        return litergewicht;
	    }


	    public String dbGetVolumenString(){
	        return volumen;
	    }

	    public String dbGetGramm100mlString() {
	    	return getEinheit("gramm").getName() + "/" + getEinheit("100ml").getName();
	    }

	  /**
	   * füllt eine Combobox mit Einheiten und selected für
	   * eine spezifizierte Einheit
	   */
	    public void setEinheitenCmb(JComboBox cmb,int value){
		     cmb.removeAllItems();
		     final Object[] e = this.einheitenListe.toArray();
		     for (final Object element : e) {
		         cmb.addItem(element);
		     }
		     try {
			     cmb.setSelectedIndex(0);
			     for(int i = 0 ;i < cmb.getModel().getSize();i++){
			         if(((EinheitsItem)cmb.getModel().getElementAt(i)).getID() == value){
			             cmb.setSelectedIndex(i);
			             break;
			         }
			     }
		     } catch (final Exception ex){}
	  }
	  /** füllt eine Combobox mit den Grundeinheiten und selected
	  * eine spezifizierte Einheit */
	 public void setGrundeinheitenCmb(JComboBox cmb,int value){
		     cmb.removeAllItems();
		     final Object[] e = grundEinheitenListe.toArray();
		     for (final Object element : e) {
		         cmb.addItem(element);
		     }
		     try {
			     cmb.setSelectedIndex(0);
			     if(value == -1) {
					value = rz.getOptionFactory().getOption("rezeptur.default_einheit",1);
				}
			     for(int i = 0 ;i < cmb.getModel().getSize();i++){
			         if(((EinheitsItem)cmb.getModel().getElementAt(i)).getID() == value){
			             cmb.setSelectedIndex(i);
			             break;
			         }
			     }
		     } catch (final Exception ex){}
	  }

	 /** getEinheit gibt die einheit mit der ID aID zurück
	  *
	  * @param aID
	  * @return EinheitsItem
	  */
	 public EinheitsItem getEinheit(String aID){
	        return (EinheitsItem)einheiten.get(aID.toLowerCase());
	 }
	 public EinheitsItem getEinheit(int aID){
	        return (EinheitsItem)einheiten.get(String.valueOf(aID));
	 }

	 /** getGrundEinheit gibt die Grundeinheit aID zurück
	  *
	  * @return EinheitsItem
	  */
	 public EinheitsItem getGrundeinheit(String aID){
	    return (EinheitsItem)grundeinheiten.get(aID);
	}

	 public EinheitsItem getGrundEinheit(int id){
		 EinheitsItem ret = null;
		 switch (id) {
			 case RzPro.EINHEIT_LITER:
				 ret = liter;
				 break;
			 case RzPro.EINHEIT_KG:
				 ret=kg;
				 break;
			 case RzPro.EINHEIT_LA:
				 ret=la;
				 break;
		 }
		 return ret;
	 }
	 public EinheitsItem getGrundeinheit(int aID) {
	     for(int i = 0; i < grundEinheitenListe.size();i++) {
	         if(((EinheitsItem)grundEinheitenListe.get(i)).getID() == aID) {
	             return (EinheitsItem)grundEinheitenListe.get(i);
	         }
	     }
	     return null;
	 }

	 /** ArrayList getEinheitenListe() gibt die EinheitenListe zurück
	  *
	  * @return ArrayList EinheitenListe
	  */
	 public ArrayList getEinheitenListe(){return einheitenListe;}
	 public ArrayList getGrundeinheitenListe(){return grundEinheitenListe;}

	 /** String getEinheitenRegex() liefert eine Regular Expression um alle gültigen Einheiten zu erkennen
	  */
	 public String getEinheitenRegex(){
	     String ret = "(";
	     for (int i=0;i < einheitenListe.size(); i++ ) {
			ret += ((EinheitsItem)einheitenListe.get(i)).getName() + "|";
		}
	     ret = ret.substring(0, ret.length() - 1);
	     return ret + "){1}";
	 }
	 /** boolean isEinheit(String) prüft, ob es sich bei dem String um eine
	  * gültige Einheit handelt
	  */
	 public boolean isEinheit(String aString){
	     einheitenMatcher = einheit.matcher(aString.toLowerCase());
	     return einheitenMatcher.matches();
	 }

	 public EinheitsItem getLiterItem() {
	     return liter;
	 }

	 public EinheitsItem getKgItem() {
	     return kg;
	 }

	 public EinheitsItem getLaItem() {
	     return la;
	 }

	 public double toLiter(double value,double staerke,double litergewicht,EinheitsItem source) {
	     return convert(value,staerke,litergewicht,source,getLiterItem());
	 }

	 public double toKg(double value,double staerke,double litergewicht,EinheitsItem source) {
	     return convert(value,staerke,litergewicht,source,getKgItem());
	 }

	 public double toLa(double value,double staerke,double litergewicht,EinheitsItem source) {
	     return convert(value,staerke,litergewicht,source,getLaItem());
	 }


	 
	 
	public double convert(double value, double staerke,double litergewicht,EinheitsItem source,EinheitsItem dest) {
	  double ret = 0.0;
	  value = value * (source.getFaktor());
      switch(dest.getTyp()) {
	  	case RzPro.EINHEIT_LITER:
	  	    switch(source.getTyp()) {
	  	    	case RzPro.EINHEIT_LITER:
	  	    	    ret = value;
	  	    		break;
	  	    	case RzPro.EINHEIT_KG:
	  	    	    ret =value * (1/litergewicht);
	  	    	    break;
	  	        case RzPro.EINHEIT_LA:
	  	            if(!rz.isZero(staerke)) {
	  	                ret = (value  * 100)/staerke; //hier!
	  	            } else {
	  	                ret = 0.0;
	  	            }
	  	            break;
	  	    }
	  		break;
	  	case RzPro.EINHEIT_KG:
	  	    switch(source.getTyp()) {
  	    	case RzPro.EINHEIT_LITER:
  	    	    ret = value  * litergewicht;
  	    		break;
  	    	case RzPro.EINHEIT_KG:
  	    	    ret = value ;
  	    	    break;
  	        case RzPro.EINHEIT_LA:
  	            if(!rz.isZero(staerke)) {
  	            ret = ((value * 100)/ staerke)*litergewicht;
  	            } else {
  	                ret =0.0;
  	            }
  	            break;
  	    }
	  		break;
	  	case RzPro.EINHEIT_LA:
	  	    switch(source.getTyp()) {
  	    	case RzPro.EINHEIT_LITER:
  	    	    if(!rz.isZero(staerke)) {
  	    	        ret = (value  * staerke)/100;
  	    	    } else {
  	    	        ret = 0.0;
  	    	    }
  	    		break;
  	    	case RzPro.EINHEIT_KG:
  	    	    if(!rz.isZero(staerke)) {
  	    	        ret = ((value  * (1/litergewicht)) * staerke)/100;
  	    	    } else {
  	    	        ret = 0.0;
  	    	    }
  	    	    break;
  	        case RzPro.EINHEIT_LA:
  	            ret=value;
  	            break;
  	    }
	  	break;
      }
      return ret * 1/dest.getFaktor();
	}

}
