package de.lunqual.rzpro.database;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.text.SimpleDateFormat;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.log.LogFactory;

import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;

import de.lunqual.rzpro.items.adressen.AdressItem;
import de.lunqual.rzpro.items.documents.DocumentHeaderItem;
import de.lunqual.rzpro.items.documents.DocumentHeaderListe;
import de.lunqual.rzpro.items.documents.DocumentItem;

public class DBDocuments {


	public static final int				TYP_NONE 	= 0;
	public static final int				TYP_PDF 		= 1;
	public static final int				TYP_JPG			= 2;
	public static final int				TYP_PNG			= 3;
	public static final int				MAX_FILE_SIZE = 16000000;
	public static final String			TYP_NONE_STRING = "";	
	public static final String			TYP_PDF_STRING = "PDF Dokument";
	public static final String			TYP_JPG_STRING = "Bild Dokument (JPG)";
	public static final String			TYP_PNG_STRING = "Bild Dokument (PNG)";
	
    RzPro           		rz;
    Connection      	con;
    DBFactory       	db;

    PreparedStatement		getAllHeaders;
    PreparedStatement		getHeadersAlle;
    PreparedStatement		getHeadersRezeptur;
    PreparedStatement 		getDocument;
    PreparedStatement		saveDocument;
    PreparedStatement		updateDocument;
    PreparedStatement 		deleteDocument;
    PreparedStatement		getDocumentListeText;
    PreparedStatement		getAllHeadersFilter;
    PreparedStatement		updateScale;
    
    /** Creates a new instance of DBBestellung */
    public DBDocuments(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
    }


    
    private void setStatements(){
        try{
	           getAllHeaders  	= con.prepareStatement("SELECT  id,typ,rezeptur,name,filename,bemerkungen,erstellt,geaendert,user1,user2 FROM " +DBFactory.TABLE_DOCUMENTS + " order by erstellt desc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	           getHeadersAlle  	= con.prepareStatement("SELECT  id,typ,rezeptur,name,filename,bemerkungen,erstellt,geaendert,user1,user2 FROM " +DBFactory.TABLE_DOCUMENTS + " where rezeptur in (?) order by erstellt desc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	           getHeadersRezeptur   	= con.prepareStatement("SELECT  id,typ,rezeptur,name,filename,bemerkungen,erstellt,geaendert,user1,user2,zoom FROM " +DBFactory.TABLE_DOCUMENTS + " where rezeptur=? order by erstellt desc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	           getDocument   	= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_DOCUMENTS + " where id=? order by erstellt desc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	           
	           saveDocument  = con.prepareStatement("INSERT INTO " + DBFactory.TABLE_DOCUMENTS  + "  (typ, rezeptur,name,filename,bemerkungen,daten,erstellt,geaendert,user1,user2,pages,zoom)  VALUES (?,?,?,?,?,?,?,?,?,?,?,?) ", ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
	           updateDocument  = con.prepareStatement("UPDATE " + DBFactory.TABLE_DOCUMENTS  + "  set typ=?, rezeptur=?,name=?,filename=?,bemerkungen=?,daten=?,geaendert=?,user2=?,pages=?,zoom=? where id=?", ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
	           deleteDocument  = con.prepareStatement("DELETE FROM  " + DBFactory.TABLE_DOCUMENTS  + "  where id=?", ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
	           getDocumentListeText   	= con.prepareStatement("SELECT name,erstellt FROM " +DBFactory.TABLE_DOCUMENTS + " where rezeptur=? order by erstellt desc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	           getAllHeadersFilter  	= con.prepareStatement("SELECT  id,typ,rezeptur,name,filename,bemerkungen,erstellt,geaendert,user1,user2 FROM " +DBFactory.TABLE_DOCUMENTS + " where name regexp(?) order by name asc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	           updateScale  = con.prepareStatement("UPDATE " + DBFactory.TABLE_DOCUMENTS  + " set zoom=?  where id=?", ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

        }catch (final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.setStatements", e.getLocalizedMessage());
        }
    }
    
    public void udpateZoom(int id,int zoom) {
    	try{
    			updateScale.setInt(1, zoom);
    			updateScale.setInt(2, id);
    			updateScale.executeUpdate(); 
    	} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.deleteDocument", e.getLocalizedMessage());
		}
    }

    public String getDocumentListeText(int rezeptur,boolean html) {
 	   	final SimpleDateFormat df = new SimpleDateFormat("EEEE dd.MM YYYY");
    	String ret = "";
    	try {
    		if(rezeptur != 0) {
    			getDocumentListeText.setInt(1,rezeptur);
    			final ResultSet rs = getDocumentListeText.executeQuery();
    			while(rs.next()) {
    				ret += rs.getString("name") + " ( " +df.format(rs.getDate("erstellt"))+ ")" + (html?"<br>":"");
    			}
    		}
    	} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getDocumentListeHTML", e.getLocalizedMessage());
		}
    	return ret;
    }
    
  
    
    public String getDokumentString(int typ) {
    	String ret = TYP_NONE_STRING;
    	switch (typ) {
    		case TYP_PDF: 		ret = TYP_PDF_STRING;break;
    		case TYP_PNG:	 	ret = TYP_PNG_STRING;break;
    		case TYP_JPG:  		ret = TYP_JPG_STRING;break;
    	}
    	return ret;
    }
    
    public void deleteDocument(int id) {
    	try{
    		if(id != 0) {
    			deleteDocument.setInt(1, id);
    			deleteDocument.executeUpdate(); 
    		}
    	} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.deleteDocument", e.getLocalizedMessage());
		}
    }
    
    public int saveDocument(DocumentItem di) {
    	int ret = 0;
    	int pages = 0;
    	if(di.getTyp() == TYP_PDF) {
    		try {
				PDDocument pdfDocument = Loader.loadPDF(di.getDaten());
				pages = pdfDocument.getNumberOfPages();
				pdfDocument.close();
			} catch (IOException e) {}
    	}
    	PreparedStatement stm;
    	try {
    		if(di.getId() == 0) {
    			stm = saveDocument;
    			stm.setInt(1, di.getTyp());
    			stm.setInt(2,di.getRezeptur());
    			stm.setString(3, di.getName());
    			stm.setString(4, di.getFilename());
    			stm.setString(5, di.getBemerkungen());
    			stm.setBytes(6,di.getDaten());
                stm.setString(7,db.dbGetDateFormatString(di.getErstellt()));
                stm.setString(8,db.dbGetDateFormatString(di.getGeaendert()));
    			stm.setString(9, db.dbGetUser());
    			stm.setString(10, db.dbGetUser());
    			stm.setInt(11,pages);
    			stm.setInt(12, di.getZoom());
    		} else {
    			stm = updateDocument;
    			stm.setInt(1,di.getTyp());
    			stm.setInt(2,di.getRezeptur());
    			stm.setString(3, di.getName());
    			stm.setString(4, di.getFilename());
    			stm.setString(5, di.getBemerkungen());
    			stm.setBytes(6,di.getDaten());
                stm.setString(7,db.dbGetDateFormatString(db.getServerTimestamp()));
    			stm.setString(8, db.dbGetUser());
    			stm.setInt(9,pages);
    			stm.setInt(10, di.getZoom());
    			stm.setInt(11, di.getId());
    		}
    		 stm.executeUpdate();
                 if(di.getId() == 0){
                     ret = db.dbGetLastInsertID("" + DBFactory.TABLE_DOCUMENTS + "");
                 	di.setId(ret);
                 }
                 else {
                     ret = di.getId();
                 }
    	} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.saveDocument", e.getLocalizedMessage());
		}
    	return ret;
    }
    
    public DocumentHeaderListe getAdressenHeaders(AdressItem ai){
    	DocumentHeaderListe  ret = new DocumentHeaderListe();
    	try {
    		if(ai != null) {
    			String liste = rz.getDatabase().getAdressen().getAdressenRezepturenliste(ai);
    			if(!liste.trim().isEmpty()) {
    				ret = this.getRezepturAlleHeaders(liste);
    			}
    		}
		} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getAdressenHeaders", e.getLocalizedMessage());
		}
    	return ret;
    }
    
    public DocumentHeaderListe getRezepturHeaders(int rezeptur){
    	DocumentHeaderListe  ret = new DocumentHeaderListe();
    	ResultSet rs = null;
    	try {
    		if(rezeptur != 0) {
    			getHeadersRezeptur.setInt(1, rezeptur);
    			rs = getHeadersRezeptur.executeQuery();
    		} else {
    			rs = getAllHeaders.executeQuery();
    		}
    		if(rs != null) {
    			while(rs.next()) {
    				ret.add(getHeaderFromRs(rs));
    			}
    		}
		} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getRezepturHeaders", e.getLocalizedMessage());
		}
    	return ret;
    }
  
    public DocumentHeaderListe getRezepturHeaders(String filter) {
    	DocumentHeaderListe  ret = new DocumentHeaderListe();
    	ResultSet rs = null;
    	try {
    		getAllHeadersFilter.setString(1, filter);
    		rs = getAllHeadersFilter.executeQuery();
    			while(rs.next()) {
    				ret.add(getHeaderFromRs(rs));
    		}
		} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getAllHeadersFilter", e.getLocalizedMessage());
		}
    	return ret;
    }
    
    public DocumentHeaderListe getRezepturAlleHeaders(String liste){
    	DocumentHeaderListe  ret = new DocumentHeaderListe ();
    	ResultSet rs = null;
    	try {
    		Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		String sql = "SELECT  id,typ,rezeptur,name,filename,bemerkungen,erstellt,geaendert,user1,user2 FROM documents where rezeptur in (" + liste +") order by erstellt desc ";
    		rs = stm.executeQuery(sql);
   			while(rs.next()) {
   				ret.add(getHeaderFromRs(rs));
   			}
		} catch (final Exception e) {
		     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getRezepturHeaders", e.getLocalizedMessage());
		}
    	return ret;
    }
    
    
    private DocumentHeaderItem getHeaderFromRs(ResultSet rs) {
    	DocumentHeaderItem ret = null;
    	try {
    		ret = new DocumentHeaderItem(
    			rs.getInt("id"),
    			rs.getInt("typ"),
    			rs.getInt("rezeptur"),
    			rs.getString("name"),
    			rs.getString("filename"),
    			rs.getString("bemerkungen"),
    			db.getDateFromString(rs.getString("erstellt")),
    			db.getDateFromString(rs.getString("geaendert")),
    			rs.getString("user2"),
    			rs.getString("user2")
    			);
    	} catch (final Exception e) {
    	     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getHeaderFromRS", e.getLocalizedMessage());
    	}
    	return ret;
    }

    private String getExtension(File f) {
    	String ret ="";
    	String fileName = f.getName();
    	int index = fileName.lastIndexOf('.');
        if(index > 0) {
        	ret = fileName.substring(index + 1);
    	}
    	return ret.toLowerCase();
    }
    
    public DocumentItem getNewDocument(int rezeptur,String path) {
    	int pages = 0;
    	DocumentItem    ret = null;
    	File f = new File(path);
    	String ext = getExtension(f);
    	int typ = 0;
    	switch(ext) {
    		case "pdf": typ = TYP_PDF; break;
    		case "jpg": typ = TYP_JPG; break;
    		case "png": typ = TYP_PNG; break;
    	}
    	 try {
    		 byte[] daten =  Files.readAllBytes(Paths.get(path));
    		 try {
    				//PDDocument pdfDocument = PDDocument.load(daten);
    				PDDocument pdfDocument = Loader.loadPDF(daten);
    				pages = pdfDocument.getNumberOfPages();
    				pdfDocument.close();
    			} catch (IOException e) {}
    		 ret = new DocumentItem(
					 0,
					 typ,
					 rezeptur,
					 f.getName(),
					 f.getName(),
					 rz.getDatabase().getRezeptur().dbGetRezepturNameNEX(rezeptur),
					daten,
					 db.getServerTimestamp(),
					 db.getServerTimestamp(),
					 db.dbGetUser(),
					 db.dbGetUser(),
					 pages,
					 100
					 );
    	} catch (IOException e) {
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getNewDocument", e.getLocalizedMessage());
		}
    	return ret;
    }
    
    public DocumentItem getDocument(int id) {
    	DocumentItem    ret = null;
    	try {
    		if(id == 0) {
    			 ret = new DocumentItem(
    					 0,
    					 TYP_NONE,
    					 0,
    					 "",
    					 "",
    					 "",
    					 new byte[0],
    					 db.getServerTimestamp(),
    					 db.getServerTimestamp(),
    					 db.dbGetUser(),
    					 db.dbGetUser(),
    					 0,
    					 100
    					 );
    		} else { 
    			getDocument.setInt(1,id);
    			ResultSet rs = getDocument.executeQuery();
    			if (rs.next()){
		    		ret = new DocumentItem(
		        			rs.getInt("documents.id"),
		        			rs.getInt("documents.typ"),
		        			rs.getInt("documents.rezeptur"),
		        			rs.getString("documents.name"),
		        			rs.getString("documents.filename"),
		        			rs.getString("documents.bemerkungen"),
		        			rs.getBlob("documents.daten").getBytes(1, (int) rs.getBlob("documents.daten").length()),
		        			db.getDateFromString(rs.getString("documents.erstellt")),
		        			db.getDateFromString(rs.getString("documents.geaendert")),
		        			rs.getString("documents.user2"),
		        			rs.getString("documents.user2"),
		        			rs.getInt("documents.pages"),
		        			rs.getInt("documents.zoom")
		        			);
		    		}
	    		}
        	} catch (final Exception e) {
        	     rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBDocuments.getDocument", e.getLocalizedMessage());
        	     ret = null;
        	} 
    	
    	return ret;
    }


}
