/*
 *  Copyright (C) 2004/2005 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.sql.Connection;
import java.util.Calendar;
import java.util.Date;


import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.items.rezeptur.ChangesFilterItem;
import de.lunqual.rzpro.items.rezeptur.ChangesFilterListe;
import de.lunqual.rzpro.items.rezeptur.ChangesItem;
import de.lunqual.rzpro.items.rezeptur.ChangesListe;
import de.lunqual.rzpro.items.stichworte.StichwortListe;
import de.lunqual.rzpro.log.LogFactory;




/**
 *
 * @author  lunqual
 */
public class DBChanges{

    RzPro           		rz;
    Connection      	con;
    DBFactory       	db;
    DialogItem			dlgItem;

    PreparedStatement		saveItem;
    PreparedStatement		getAuswahlAlle;
    PreparedStatement		getAuswahlRelevante;
    PreparedStatement		isRelevant;
    PreparedStatement		isChanges;
    
    PreparedStatement		getListeAlle;
    PreparedStatement		getListeDatum;
    PreparedStatement		getListeUser;
    PreparedStatement		getListeDatumUser;
    PreparedStatement		deleteChanges;
    PreparedStatement		hasRelevantChanges;
    
    PreparedStatement		getLetzteBuchung;
    PreparedStatement		setIrRelevant;
    PreparedStatement		setIrRelevantAlle;
    
    /** Creates a new instance of DBLager */
    public DBChanges(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
    }

    private void setStatements(){
        try{
            saveItem = con.prepareStatement("INSERT INTO " + DBFactory.TABLE_CHANGES + " (rezeptur_id,bezeichnung,alt,neu,erstellt,user,relevant) VALUES (?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE ); 
            getAuswahlAlle =  con.prepareStatement("SELECT changes.id,changes.rezeptur_id,changes.bezeichnung,alt,neu,changes.erstellt,changes.user,rezeptliste.name,rezeptliste.stichworte,changes.relevant FROM changes left join rezeptliste on changes.rezeptur_id=rezeptliste.id WHERE changes.rezeptur_id=? ORDER BY changes.erstellt desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getAuswahlRelevante =  con.prepareStatement("SELECT changes.id,changes.rezeptur_id,changes.bezeichnung,alt,neu,changes.erstellt,changes.user,rezeptliste.name,rezeptliste.stichworte,changes.relevant FROM changes left join rezeptliste on changes.rezeptur_id=rezeptliste.id  WHERE changes.rezeptur_id=? AND changes.relevant <> 0 ORDER BY changes.erstellt desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            isRelevant = con.prepareStatement("SELECT id from buchungen WHERE rezeptur_id=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            isChanges = con.prepareStatement("SELECT id from changes WHERE rezeptur_id=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            deleteChanges = con.prepareStatement("DELETE FROM " + DBFactory.TABLE_CHANGES + " where rezeptur_id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE ); 
            hasRelevantChanges = con.prepareStatement("SELECT id from changes where rezeptur_id=? and relevant <> 0 LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLetzteBuchung = con.prepareStatement("SELECT id,datum from buchungen WHERE rezeptur_id=? order by id asc LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            setIrRelevant = con.prepareStatement("UPDATE " + DBFactory.TABLE_CHANGES + " set relevant=0 where rezeptur_id=? AND erstellt <=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE ); 
            setIrRelevantAlle = con.prepareStatement("UPDATE " + DBFactory.TABLE_CHANGES + " set relevant=0 where rezeptur_id=?  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE ); 
            
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAcid.setStatements", e.getLocalizedMessage());
        }
    }

    public ChangesFilterListe getNames() {
    	String[] n = new String[]{
    		rz.getLocale().getString("changes.titel"),
    		rz.getLocale().getString("changes.rezeptur"),
    		rz.getLocale().getString("changes.staerke"),
    		rz.getLocale().getString("changes.basismenge"),
    		rz.getLocale().getString("changes.basiseinheit"),
    		rz.getLocale().getString("changes.extrakt"),
    		rz.getLocale().getString("changes.litergewicht"),
    		rz.getLocale().getString("changes.extraktfrei"),
    		rz.getLocale().getString("changes.adressliste"),
    		rz.getLocale().getString("changes.stichwortliste"),
    		rz.getLocale().getString("changes.deklarationsliste"),
    		rz.getLocale().getString("changes.lager"),
    		rz.getLocale().getString("changes.artikelnummer"),
    		rz.getLocale().getString("changes.losnummer"),
    		rz.getLocale().getString("changes.meldebestand"),
    		rz.getLocale().getString("changes.vorschlagsbestand"),
    		rz.getLocale().getString("changes.bestandseinheit"),
    		rz.getLocale().getString("changes.preismenge"),
    		rz.getLocale().getString("changes.preis"),
    		rz.getLocale().getString("changes.preiseinheit"),
    		rz.getLocale().getString("changes.waehrung"),
    		rz.getLocale().getString("changes.schwundsatz"),
    		rz.getLocale().getString("changes.brennwert_af"),
    		rz.getLocale().getString("changes.acid"),
    		rz.getLocale().getString("changes.tank"),
    		rz.getLocale().getString("changes.spezifikation"),
    		rz.getLocale().getString("changes.rprobe"),
    		rz.getLocale().getString("changes.tempvon"),
    		rz.getLocale().getString("changes.tempbis"),
    		rz.getLocale().getString("changes.konform"),    	
    		rz.getLocale().getString("changes.spezifikation_text"),
    		rz.getLocale().getString("changes.ean"),
    		rz.getLocale().getString("changes.steuerlager"),
    		rz.getLocale().getString("changes.sorte"),
    		rz.getLocale().getString("changes.autoanfrage"),
    		rz.getLocale().getString("changes.probenanzahl"),
    		rz.getLocale().getString("changes.mhd"),
    		rz.getLocale().getString("changes.laborwert"),
    	};
    	Arrays.sort(n);
    	ChangesFilterListe ret = new ChangesFilterListe(rz);
    	for(int i=0;i < n.length;i++ ) {
    		ret.add(new ChangesFilterItem(true, n[i],i));
    	}
    	return ret;
    }
    
    private void setIrRelevant(int rezeptur_id,Date datum) {
    	if(rezeptur_id != 0) {
	    	try{
	    		if(datum == null) {
	    			setIrRelevantAlle.setInt(1, rezeptur_id);
	    			setIrRelevantAlle.executeUpdate();
	    		}else {
	    			setIrRelevant.setInt(1, rezeptur_id);
	    			setIrRelevant.setString(2, db.dbGetDateFormatString(datum));
	    			setIrRelevant.executeUpdate();
	    		}
	        }
	        catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.setRelevant", e.getLocalizedMessage());
	        }
    	}	
    }
    
    public void cleanupAfterStorno(int rezeptur_id) {
    	if(rezeptur_id != 0) {
	    	try{
	    		getLetzteBuchung.setInt(1,rezeptur_id);
	    		ResultSet rs = getLetzteBuchung.executeQuery();
	    		if(rs.next()) {
	    			// letzte Buchung gefunden
	    			setIrRelevant(rezeptur_id,db.getDateFromString(rs.getString("datum")));
	    		} else {
	    			// Keine Buchungen vorhanden...
	    			setIrRelevant(rezeptur_id,null);
	    		}
	        }
	        catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.cleanUpAfterStorno", e.getLocalizedMessage());
	        }
    	}
    }
    
    public void delete(int rezeptur_id) {
    	if(rezeptur_id != 0) {
	    	try{
	    		deleteChanges.setInt(1,rezeptur_id);
	    		deleteChanges.executeUpdate();
	        }
	        catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.delete", e.getLocalizedMessage());
	        }
    	}
    }
    
    public boolean commit(ChangesListe liste) {
    	boolean ret = true;
    	if(liste.getRezepturID() != 0) {
	    	try{
	    		for(int i=0;i < liste.getListe().size();i++) {
	    			ChangesItem c = liste.getListe().get(i);
	    			saveItem.setInt(1,liste.getRezepturID());
	    			saveItem.setString(2,c.getBezeichnung());
	    			saveItem.setString(3,c.getAlt());
	    			saveItem.setString(4,c.getNeu());
	        		saveItem.setString(5,db.dbGetDateFormatString(db.getServerTimestamp()) );
	        		saveItem.setString(6,db.dbGetUser());
	        		saveItem.setBoolean(7, c.isRelevant());
	        		saveItem.executeUpdate();
	    		}
	        }
	        catch (final Exception e){
	        	ret = false;
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.commit", e.getLocalizedMessage());
	        }
    	}
    	return ret;
    }

	private String getChangesWhereClause(Date von, Date bis,String benutzer,String rezeptur,boolean relevant) {
        String ret = "" ;
        if((von != null) && (bis != null)) {
        	SimpleDateFormat df=new SimpleDateFormat("yyyy-MM-dd");
            ret += " AND changes.erstellt BETWEEN '" + df.format(von) + " 00:00:00' AND '" + df.format(bis) + " 23:59:59' ";
        }
        if(benutzer != null) {
            if(!benutzer.equals("")) {
				ret += " AND user REGEXP '" + benutzer.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(rezeptur != null) {
            if(!rezeptur.equals("")) {
				ret += " AND rezeptliste.name REGEXP '" + rezeptur.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(relevant) {
        	ret += " AND changes.relevant <> 0 ";
        }
         if(ret.startsWith(" AND")) {
        	ret = ret.substring(4);
        }
        return ret ;
    }
    
	private boolean isOutputAllowed(ChangesItem ci, ChangesFilterListe filter) {
		boolean ret = false;
		
		return ret;
	}
	
	
    public ArrayList<ChangesItem> getChanges(Date von,Date bis,String user,String rezeptur,boolean relevante) {
    	ArrayList<ChangesItem> ret = new ArrayList<ChangesItem>();
    	try {
            final Statement stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            final String clause=getChangesWhereClause(von,bis,user,rezeptur,relevante);
            final String sql = "SELECT changes.id,changes.rezeptur_id,changes.bezeichnung,alt,neu,changes.erstellt,changes.user,rezeptliste.name,rezeptliste.stichworte,changes.relevant FROM " +
            		"changes left join rezeptliste on changes.rezeptur_id=rezeptliste.id "+ 
            		 (clause.equals("")?" ":" WHERE" + clause) +
            		" ORDER BY changes.id DESC";
            final ResultSet rs = stm.executeQuery(sql);
        	while(rs.next()) {
        		if(rs.getString("rezeptliste.name") != null && rs.getString("rezeptliste.stichworte") != null) {
	        		ChangesItem ci = new  ChangesItem(
	        				rz,
	        				rs.getInt("changes.id"),
	        				rs.getInt("rezeptur_id"),
	        				rs.getString("changes.bezeichnung"),
	        				rs.getString("alt"),
	        				rs.getString("neu"),
	        				db.getDateFromString(rs.getString("changes.erstellt")),
	        				rs.getString("changes.user"),
	        				(rs.getString("rezeptliste.name") != null)?rs.getString("rezeptliste.name"):"",
	        				(rs.getString("rezeptliste.stichworte") != null) ?db.getSTW().dbGetStichwortListe(rs.getString("rezeptliste.stichworte")):new StichwortListe(),
	        				rs.getBoolean("relevant")
	        				);
	        		   ret.add(ci);
        		}
        	}
		}   catch (final Exception e){
	        rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.getChanges - Liste", e.getLocalizedMessage());
		}
    	
    	return ret;
    }
    
    public ArrayList<ChangesItem> getChanges(int id,boolean relevante) {
    	ArrayList<ChangesItem> ret = new ArrayList<ChangesItem>();
    	PreparedStatement stm=null;
    	try {
	    	if(relevante) {
	    		stm=getAuswahlRelevante;
	    	}else {
	    		stm=getAuswahlAlle;
	    	}
	    	if(stm != null) {
	    		stm.setInt(1,id);
	        	final ResultSet rs = stm.executeQuery();
	        	while(rs.next()) {
	        		if(rs.getString("rezeptliste.name") != null && rs.getString("rezeptliste.stichworte") != null) {
		        		ChangesItem ci =  new ChangesItem(
		        				rz,
		        				rs.getInt("changes.id"),
		        				rs.getInt("rezeptur_id"),
		        				rs.getString("changes.bezeichnung"),
		        				rs.getString("alt"),
		        				rs.getString("neu"),
		        				db.getDateFromString(rs.getString("changes.erstellt")),
		        				rs.getString("changes.user"),
		        				(rs.getString("rezeptliste.name") != null)?rs.getString("rezeptliste.name"):"",
		        				(rs.getString("rezeptliste.stichworte") != null) ?db.getSTW().dbGetStichwortListe(rs.getString("rezeptliste.stichworte")):new StichwortListe(),
		        				rs.getBoolean("relevant")
		        				);
			        		   ret.add(ci);
	        		}
	        	}
	    	}
    	}   catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.getChanges", e.getLocalizedMessage());
    	}
    	
    	return ret;
    }

    public boolean isChanges(int rezeptur_id) {
    	boolean ret=false;
    	if(rezeptur_id > 0) {
	    	try {
	    		isChanges.setInt(1, rezeptur_id);
	    		final ResultSet rs = isChanges.executeQuery();
	    		if(rs.next()) {
	    				ret=true;
	    		}
	    	} catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.isChanges", e.getLocalizedMessage());
	    	}
    	}
    	return ret;
    }
    
    public boolean isRelevant(int rezeptur_id) {
    	boolean ret=false;
    	if(rezeptur_id > 0) {
	    	try {
	    		isRelevant.setInt(1, rezeptur_id);
	    		final ResultSet rs = isRelevant.executeQuery();
	    		if(rs.next()) {
	    				ret=true;
	    		}
	    	} catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.isRelevant", e.getLocalizedMessage());
	    	}
    	}
    	return ret;
    }
  
    public boolean hasRelevanteAenderungen(int rezeptur_id) {
    	boolean ret=false;
    	if(rezeptur_id > 0) {
	    	try {
	    		hasRelevantChanges.setInt(1, rezeptur_id);
	    		final ResultSet rs = hasRelevantChanges.executeQuery();
	    		if(rs.next()) {
	    				ret=true;
	    		}
	    	} catch (final Exception e){
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBChanges.hasRelevanteAenderungen", e.getLocalizedMessage());
	    	}
    	}
    	return ret;
    }
    
    
    public boolean isSameDay(Calendar c1,Calendar c2) {
    	boolean ret = false;
    	if(
    			(c1.get(Calendar.YEAR) == c2.get(Calendar.YEAR)) &&
    	    	(c1.get(Calendar.MONTH) == c2.get(Calendar.MONTH)) &&
    	       (c1.get(Calendar.DAY_OF_MONTH) == c2.get(Calendar.DAY_OF_MONTH))
    	  ){
    	   		ret = true;
    	   	}
    	return ret;
    }
}
