/*
 *  Copyright (C) Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.DateFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.fenster.auswahlfenster.Auswahl;
import de.lunqual.rzpro.fenster.buchungsfenster.HerstellungsFenster;
import de.lunqual.rzpro.fenster.buchungsfenster.StatistikFenster;
import de.lunqual.rzpro.fenster.dialoge.DialogControl;
import de.lunqual.rzpro.items.mhdItem;
import de.lunqual.rzpro.items.auftrag.AuftragsItem;
import de.lunqual.rzpro.items.buchung.BuchungLosnummerItem;
import de.lunqual.rzpro.items.buchung.BuchungsItem;
import de.lunqual.rzpro.items.buchung.BuchungsListe;
import de.lunqual.rzpro.items.buchung.ClientBuchungsItem;
import de.lunqual.rzpro.items.buchung.HerstellungsItem;
import de.lunqual.rzpro.items.buchung.HerstellungsListe;
import de.lunqual.rzpro.items.buchung.LosnummerItem;
import de.lunqual.rzpro.items.buchung.LosnummerListe;
import de.lunqual.rzpro.items.buchung.SaldenItem;
import de.lunqual.rzpro.items.buchung.StatistikListenItem;
import de.lunqual.rzpro.items.buchung.StatistikRezepturItem;
import de.lunqual.rzpro.items.buchung.StatistikRezepturListe;
import de.lunqual.rzpro.items.buchung.TankLastBelegung;
import de.lunqual.rzpro.items.buchung.commentLosnummerItem;
import de.lunqual.rzpro.items.buchung.commentLosnummerListe;
import de.lunqual.rzpro.items.buchung.herstellungReturnItem;
import de.lunqual.rzpro.items.buchung.letzteVerwendungItem;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.items.einheiten.EinheitsItem;
import de.lunqual.rzpro.items.ergebnis.ErgebnisItem;
import de.lunqual.rzpro.items.ergebnis.ErgebnisListe;
import de.lunqual.rzpro.items.lager.LagerDruckListe;
import de.lunqual.rzpro.items.lager.LagerListe;
import de.lunqual.rzpro.items.lager.LagerListenItem;
import de.lunqual.rzpro.items.mischungen.MischungsItem;
import de.lunqual.rzpro.items.mischungen.TicketItem;
import de.lunqual.rzpro.items.rechnen.RechenItem;
import de.lunqual.rzpro.items.rezeptur.BestandswarnungsItem;
import de.lunqual.rzpro.items.rezeptur.RezepturBuchungsItem;
import de.lunqual.rzpro.items.rezeptur.RezepturItem;
import de.lunqual.rzpro.items.rezeptur.SteuerDatenItem;
import de.lunqual.rzpro.items.statistik.WerteItem;
import de.lunqual.rzpro.items.statistik.WerteListe;
import de.lunqual.rzpro.items.tank.tankItem;
import de.lunqual.rzpro.items.verbrauch.VerbrauchsItem;
import de.lunqual.rzpro.items.verbrauch.VerbrauchsListe;
import de.lunqual.rzpro.log.LogFactory;
import de.lunqual.rzpro.options.OptionFactory;
/**
 *
 * @author  lunqual
 */
public class DBBuchung {


    //Schwundabzug vorwärts oder rückwärts
    public static final int		BUCHUNG_HERSTELLUNG 		= 0;
    public static final int		BUCHUNG_VERSAND 				= 1;
    public static final int		BUCHUNG_HERSTELLUNG_REST 		= 2;
    public static final int		BUCHUNG_VERSAND_REST 				= 3;
    //Losnummern-entnahme FIFO=älteste zuerste, LIFO=neueste zuerst
    public static final int 	REIHE_FIFO 							= 0;
    public static final int		REIHE_LIFO							= 1;
 
    //Typ der Buchung
    public static final int		TYP_ZUGANG 						= 1;
    public static final int 		TYP_ABGANG 						= 2;

    //Code der Buchung
    public static final int		CODE_HERSTELLUNG 				= 1;
    public static final int		CODE_EINKAUF  					= 2;
    public static final int		CODE_ENTNAHME 					= 3;
    public static final int		CODE_SCHWUND					= 4;
    public static final int		CODE_SCHWUNDAUSGLEICH 	= 5;
    public static final int		CODE_KORREKTUR_ZUGANG	= 6;
    public static final int		CODE_KORREKTUR_ABGANG  = 7;

    public static final double  BUCHUNG_PRECISION           = 0.01;
    public static final int 	DEFAULT_VERBRAUCH_JAHRE=10;
    
    private static final int		MAX_DIFFERENCE = 100;
    
    //maximale größe von  buchungen.comment = 1KB !


    RzPro           		rz;
    Connection      	con;
    DBFactory       	db;
    DialogItem 		dlgItem;
    double				precision = BUCHUNG_PRECISION;

    // Statements
    PreparedStatement   	getBuchung;
    PreparedStatement   	getTemp1Buchung;
    PreparedStatement		saveBuchung;
    PreparedStatement		getLosnummerListeFIFO;
    PreparedStatement		getLosnummerListeLIFO;
//
    PreparedStatement		zugangKorrektur; // korrigiert zugangsbuchungen bei Abgängen

    PreparedStatement		checkLosnummerExists;

    PreparedStatement   	getNextLosnummer;
    PreparedStatement  		saveHerstellung;
    PreparedStatement		updateHerstellung;
    PreparedStatement		updateBuchung;
    PreparedStatement		deleteHerstellung;
    PreparedStatement     	getHerstellung;

    PreparedStatement		checkLastEntry;
    PreparedStatement    	checkLastHerstellung;

    PreparedStatement    	deleteBuchung;
    PreparedStatement    	getZugangsBuchung;
    PreparedStatement		stornoEntnahme;
    PreparedStatement		setBuchungenErledigt;
    PreparedStatement		setHerstellungErledigt;

    PreparedStatement 		getEinzelHerstellung;

    PreparedStatement		checkRezepturUsed;

    PreparedStatement		getHerstellungsLos;

    PreparedStatement		getLosnummerVerlauf0;
    PreparedStatement		getLosnummerVerlauf1;
    PreparedStatement		getLosnummerVerlauf2;

    PreparedStatement		getBestandsListeEinzeln;
    PreparedStatement		getBestandsSaldoEinzeln;
    PreparedStatement		getBestandsListeTemp;
    PreparedStatement		getBestandsSaldoTemp;
    PreparedStatement		createBestandTemp;
    PreparedStatement		moveBestandTemp;
    PreparedStatement		deleteBestandTemp;

    PreparedStatement		getLosnummerTemplate;
    PreparedStatement      	getLosnummerVorschlagTemplate;
    PreparedStatement		setLosnummerTemplate;
    PreparedStatement		setLosnummerVorschlagTemplate;

    PreparedStatement		getStatistikListe;
    PreparedStatement 		setEinzelComment;
    PreparedStatement		getEinzelComment;

    PreparedStatement    	getVerbrauchZugang;
    PreparedStatement    	getVerbrauchAbgang;

    PreparedStatement		getHerstellungsRezepturen;
    
    PreparedStatement 		getBestandTanks;
   
    PreparedStatement		getHerstellungsDaten;
    
    PreparedStatement		getRProbeErforderlich;
    PreparedStatement		getRProbeVorhanden;
    
    PreparedStatement		getBuchungenForToday;

    PreparedStatement    	getErsteBuchung;
    PreparedStatement    	getMhdDaten;
    PreparedStatement		saveMhd;
    PreparedStatement		getMhdListeMhd;
    PreparedStatement		getMhdListeRezeptur;
    PreparedStatement		getMhdListeMhdReverse;
    PreparedStatement		getMhdListeRezepturReverse;
    
    PreparedStatement		getMhdListeMhdDatum;
    PreparedStatement		getMhdListeRezepturDatum;
    PreparedStatement		getMhdListeMhdReverseDatum;
    PreparedStatement		getMhdListeRezepturReverseDatum;
    
    PreparedStatement		isRelevant;
    
    PreparedStatement		getZugangsBuchungByLosnummer;
    PreparedStatement		getLosnummerSuchListe;
    
    PreparedStatement		getLosnummerFromHerstellung;
    PreparedStatement		setHerstellungLosnummer;
    
    PreparedStatement			getRestMenge;
    PreparedStatement		getHerstellungFromLosnummer;
    
   
    
    Calendar				mhd;
    Calendar				now;
    String					monate[];
    int						max_comment_size=1024;
    
    private static final String  BUCHUNG_FIELD_LIST=" buchungen.id,"+
    	"buchungen.typ,code, buchungen.lager,herstellung,buchungen.rezeptur_id,datum,buchungen.user,buchungen.bezeichnung,"+
    	"buchungen.losnummer, losnummer_reihe,buchungen.comment, erledigt,uebernommen,"+
    	"einheit, buchungen.staerke,buchungen.litergewicht,liter,kg,buchungen.la, rest_liter, rest_kg,rest_la, rest,mhd,mischungen.id,buchungen.steuer_artikel,buchungen.sorte,buchungen.steuerlager,sorten.bezeichnung,steuer_lager.bezeichnung" ;

    // wenn eine statistik abgerufen wird, soll eine Warnung ausgegeben werden, wenn
    // mehr als SIZE_FOR_BATCH Rezepturen betroffen sind, weil das sonst sehr lange
    // dauern kann.
    public static final int	SIZE_FOR_BATCH = 25;

    /** Creates a new instance of DBBuchung */
    public DBBuchung(RzPro r,DBFactory aDb,Connection aCon) {
        rz = r;
        precision = rz.getOptionFactory().getDouble("buchung.precision",BUCHUNG_PRECISION);
        db = aDb;
        con = aCon;
        mhd=Calendar.getInstance();
        now=Calendar.getInstance();
        monate= new DateFormatSymbols().getMonths();
        now.setTime(new Date());
        setStatements();
        dlgItem = new DialogItem(0,"",0.0,"","","","","","",null);
    }

    private void setStatements(){
        try{
            getBuchung     = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST +" FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer  left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getEinzelHerstellung  = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_HERSTELLUNG + "  WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveBuchung     = con.prepareStatement("INSERT INTO " + DBFactory.TABLE_BUCHUNG + " (typ,code,lager,herstellung,rezeptur_id,datum,user,bezeichnung,losnummer,losnummer_reihe,comment,erledigt,uebernommen, " +
                    "einheit,staerke,litergewicht,liter,kg,la,rest_liter,rest_kg,rest_la,rest,mhd,steuer_artikel,sorte,steuerlager)" +
                    "VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?) ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getLosnummerListeFIFO = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.rezeptur_id=? AND typ=1 AND rest <> 0 ORDER BY datum ASC",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLosnummerListeLIFO = con.prepareStatement("SELECT "+ BUCHUNG_FIELD_LIST +" FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.rezeptur_id=? AND typ=1 AND rest <> 0 ORDER BY datum DESC",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            zugangKorrektur = con.prepareStatement("UPDATE " + DBFactory.TABLE_BUCHUNG +
                    "  set rest_liter=?,rest_kg=?,rest_la=?,rest=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

            checkLosnummerExists    = con.prepareStatement("SELECT losnummer FROM " + DBFactory.TABLE_BUCHUNG + "  WHERE losnummer=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            getNextLosnummer     = con.prepareStatement("SELECT MAX(id) as max FROM  " + DBFactory.TABLE_HERSTELLUNG ,ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveHerstellung = con.prepareStatement("INSERT INTO herstellungen (rezeptur,bezeichnung,datum,losnummer,liter,kg,la,schwundsatz,endmenge_liter,endmenge_kg,endmenge_la,user,tank,r_probe) VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

            getHerstellung     = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + " left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE herstellung=? ORDER BY typ,buchungen.id",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            checkLastEntry    = con.prepareStatement("SELECT id,rezeptur_id,datum FROM " + DBFactory.TABLE_BUCHUNG + "  WHERE rezeptur_id=? and datum >? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            deleteBuchung    = con.prepareStatement("DELETE FROM "+ DBFactory.TABLE_BUCHUNG + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getZugangsBuchung  = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.rezeptur_id=? AND  buchungen.losnummer=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            stornoEntnahme   = con.prepareStatement("UPDATE "+ DBFactory.TABLE_BUCHUNG + " set rest_liter=?,rest_kg=?,rest_la=?,rest=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            setBuchungenErledigt  = con.prepareStatement("UPDATE "+ DBFactory.TABLE_BUCHUNG + " set erledigt=? WHERE herstellung=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            setHerstellungErledigt  = con.prepareStatement("UPDATE "+ DBFactory.TABLE_HERSTELLUNG + " set erledigt=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

            updateHerstellung=con.prepareStatement("UPDATE " + DBFactory.TABLE_HERSTELLUNG + " set bezeichnung=?, schwundsatz=?,endmenge_liter=?,endmenge_kg=?,endmenge_la=?, tank=?,r_probe=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateBuchung=con.prepareStatement("UPDATE " + DBFactory.TABLE_BUCHUNG + " set bezeichnung=?,comment=?,lager=?,steuer_artikel=?,sorte=?,steuerlager=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteHerstellung =con.prepareStatement("DELETE FROM " + DBFactory.TABLE_HERSTELLUNG + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            checkLastHerstellung    = con.prepareStatement("SELECT id,rezeptur,datum FROM " + DBFactory.TABLE_HERSTELLUNG + "  WHERE rezeptur=? and datum > ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            checkRezepturUsed   = con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_BUCHUNG + "  WHERE rezeptur_id=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getHerstellungsLos    = con.prepareStatement("SELECT herstellungen.id,herstellungen.losnummer,proben.losnummer FROM " + DBFactory.TABLE_HERSTELLUNG + " left join proben on proben.losnummer=herstellungen.losnummer  WHERE herstellungen.id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getHerstellungsDaten    = con.prepareStatement("SELECT herstellungen.id,herstellungen.losnummer,herstellungen.bezeichnung,proben.id FROM " + DBFactory.TABLE_HERSTELLUNG + " left join proben on proben.losnummer=herstellungen.losnummer  WHERE herstellungen.id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            
            getLosnummerVerlauf0    = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + " left join mischungen on buchungen.losnummer=mischungen.losnummer  left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.losnummer regexp ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLosnummerVerlauf1    = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE comment regexp ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLosnummerVerlauf2    = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.losnummer regexp ? AND comment  regexp ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            
            getBestandsListeEinzeln = con.prepareStatement("SELECT " +BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.rezeptur_id=? AND rest <> 0 ORDER BY datum", ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestandsSaldoEinzeln = con.prepareStatement("SELECT sum(rest_liter) as summe_liter,sum(rest_kg) as summe_kg,sum(rest_la) as summe_la FROM " + DBFactory.TABLE_BUCHUNG + " WHERE buchungen.rezeptur_id=? AND rest <> 0 ORDER BY datum", ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLosnummerTemplate = con.prepareStatement("SELECT template FROM losnummer" , ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getLosnummerVorschlagTemplate = con.prepareStatement("SELECT vtemplate FROM losnummer" , ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            setLosnummerTemplate = con.prepareStatement("UPDATE losnummer SET template=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            setLosnummerVorschlagTemplate = con.prepareStatement("UPDATE losnummer SET vtemplate=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

            getStatistikListe = con.prepareStatement("SELECT " + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + "  left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id WHERE buchungen.losnummer=? ORDER BY datum" , ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            setEinzelComment=con.prepareStatement("UPDATE " + DBFactory.TABLE_BUCHUNG + " set comment=? where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getEinzelComment=con.prepareStatement("SELECT comment from " + DBFactory.TABLE_BUCHUNG + " where typ=? and losnummer=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            getVerbrauchZugang= con.prepareStatement("SELECT sum(liter) as sliter,sum(kg) as skg, sum(la) as sla, year(datum) as jahr FROM " + DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id=? and typ=" +String.valueOf(TYP_ZUGANG)+  "  group by year(datum) order by year(datum) desc LIMIT " + String.valueOf(DEFAULT_VERBRAUCH_JAHRE),ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getVerbrauchAbgang= con.prepareStatement("SELECT sum(liter) as sliter,sum(kg) as skg, sum(la) as sla, year(datum) as jahr FROM " + DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id=? and typ=" + String.valueOf(TYP_ABGANG) + "  group by year(datum) order by year(datum) desc LIMIT " + String.valueOf(DEFAULT_VERBRAUCH_JAHRE),ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
      
            getHerstellungsRezepturen= con.prepareStatement("SELECT  distinct rezeptur_id from " + DBFactory.TABLE_BUCHUNG + " WHERE herstellung=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

    		getBestandTanks=con.prepareStatement("SELECT mischungen.id,buchungen.id,buchungen.bezeichnung,datum,buchungen.losnummer,comment,rest_liter as liter,mhd FROM " +DBFactory.TABLE_BUCHUNG + " left join mischungen on mischungen.losnummer=buchungen.losnummer WHERE buchungen.rezeptur_id=? AND rest <> 0  ORDER BY datum",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		
    		getRProbeErforderlich=con.prepareStatement("SELECT r_probe FROM " +DBFactory.TABLE_HERSTELLUNG + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		getRProbeVorhanden=con.prepareStatement("SELECT count(id) as anzahl FROM " +DBFactory.TABLE_PROBEN + " WHERE losnummer=?  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
   
    		getBuchungenForToday=con.prepareStatement("SELECT id,datum,liter,user,bezeichnung,losnummer FROM " +DBFactory.TABLE_BUCHUNG + " WHERE user=? AND datum >=? ORDER BY ID DESC",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
        	
    		getErsteBuchung=con.prepareStatement("SELECT min(datum) as minDatum FROM " +DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id=? LIMIT 1 ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
     		getMhdDaten=con.prepareStatement("SELECT id,rezeptur_id,bezeichnung,losnummer,mhd FROM " +DBFactory.TABLE_BUCHUNG + " WHERE id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveMhd=con.prepareStatement("UPDATE " + DBFactory.TABLE_BUCHUNG + " set mhd=? where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            String join = "SELECT buchungen.id,rezeptur_id,rezeptliste.name as bezeichnung,buchungen.losnummer as losnummer,mhd,rezeptliste.artikelnummer as artikelnummer,rezeptliste.tempvon as tempvon,rezeptliste.tempbis as tempbis FROM " +DBFactory.TABLE_BUCHUNG;
            join += " left join " + DBFactory.TABLE_REZEPTLISTE +" on buchungen.rezeptur_id=rezeptliste.id ";
            getMhdListeMhd=con.prepareStatement(join+ " WHERE rest<>0 AND mhd <> 0 ORDER BY mhd ASC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		getMhdListeRezeptur=con.prepareStatement(join + " WHERE rest<>0 AND mhd <> 0 ORDER BY bezeichnung ASC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
     		getMhdListeMhdReverse=con.prepareStatement(join + " WHERE rest<>0 AND mhd <> 0 ORDER BY mhd DESC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		getMhdListeRezepturReverse=con.prepareStatement(join + " WHERE rest<>0 AND mhd <> 0 ORDER BY bezeichnung DESC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);



            getMhdListeMhdDatum=con.prepareStatement(join+ " WHERE rest<>0 AND  mhd <> 0 AND mhd >=? AND mhd <= ? ORDER BY mhd ASC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		getMhdListeRezepturDatum=con.prepareStatement(join + " WHERE rest<>0  AND mhd <> 0 AND  mhd >=? AND mhd <= ? ORDER BY bezeichnung ASC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
     		getMhdListeMhdReverseDatum=con.prepareStatement(join + " WHERE rest<>0  AND mhd <> 0 AND  mhd >=? AND mhd <= ? ORDER BY mhd DESC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		getMhdListeRezepturReverseDatum=con.prepareStatement(join + " WHERE rest<>0 AND mhd <> 0 AND mhd >=? AND mhd <= ? ORDER BY bezeichnung DESC  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

    		

       		getZugangsBuchungByLosnummer=con.prepareStatement("select bezeichnung,herstellung FROM "  + DBFactory.TABLE_BUCHUNG + " WHERE typ=" + TYP_ZUGANG + "  AND losnummer=? LIMIT 1  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
       		getLosnummerSuchListe=con.prepareStatement("select rezeptur_id,id,bezeichnung,losnummer,liter,datum FROM "  + DBFactory.TABLE_BUCHUNG + " WHERE typ=" + TYP_ZUGANG + "  AND losnummer regexp ? order by datum desc LIMIT 9 ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		getLosnummerFromHerstellung=con.prepareStatement("select losnummer FROM "  + DBFactory.TABLE_HERSTELLUNG + " WHERE id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		setHerstellungLosnummer=con.prepareStatement("update "  + DBFactory.TABLE_HERSTELLUNG + " set losnummer=? WHERE id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

       		getRestMenge=con.prepareStatement("select id,rest_liter,rest_kg,rest_la,losnummer FROM "  + DBFactory.TABLE_BUCHUNG + " WHERE typ=" + TYP_ZUGANG + "  AND losnummer=? LIMIT 1  ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
       		
            getHerstellungFromLosnummer     = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_HERSTELLUNG + " WHERE losnummer regexp ? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);


            
            getMaxCommentSizeOnce();
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAbfrage.setStatements", e.getLocalizedMessage());
        }
    }

    public ArrayList<TankLastBelegung> getTankLastBelegung(tankItem ti,int count){
    	ArrayList<TankLastBelegung>ret = new ArrayList<TankLastBelegung>();
    	if(!ti.getNummer().trim().equals("") && count > 0) {
	    	try {
	    		if(ti.isMischung()) {
	    			String sql = "SELECT id,bezeichnung,erstellt from " + DBFactory.TABLE_MISCHUNGEN  +
	    					" where tank='" + ti.getNummer().trim() + "' " +
							" order by id desc limit " + String.valueOf(count);
		    		Statement stm = con.createStatement();
		    		ResultSet rs = stm.executeQuery(sql);
		    		while (rs.next()) {
			    			ret.add(new TankLastBelegung(
			    					rs.getInt("id"),
			    					rs.getString("bezeichnung"),
			    					db.getDateFromString(rs.getString("erstellt")),
			    					0,
			    					String.valueOf(rs.getInt("id")),
			    					rs.getInt("id")
			    					));
			    		}
	    		} else {
		    		String sql = "SELECT id,bezeichnung,datum,liter,losnummer,comment,rezeptur_id from " + DBFactory.TABLE_BUCHUNG + " where typ=1 and " +
		    				"comment regexp '" + rz.getLocale().getString("dbbuchung.tank_last_regex").replaceAll("%s", ti.getNummer().trim()) + "' " + 
							" order by id desc";

		    		int limit =0;
		    		Statement stm = con.createStatement();
		    		ResultSet rs = stm.executeQuery(sql);
		    		while (rs.next() && limit < count) {
		    			//String c = rz.getDatabase().getTank().getTankString(rs.getString("comment"));
		    			if(tankNummerExclusive(rs.getString("comment").trim(),ti.getNummer().trim())){
		    				limit++;
			    			ret.add(new TankLastBelegung(
			    					rs.getInt("id"),
			    					rs.getString("bezeichnung"),
			    					db.getDateFromString(rs.getString("datum")),
			    					rs.getDouble("liter"),
			    					rs.getString("losnummer"),
			    					rs.getInt("rezeptur_id")
			    					));
			    			}
			    		}
	    		}
	    				
	    	}catch (final Exception e){
				rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchungen.getTankLastBelegung" , e.getLocalizedMessage());
			}
    	}
    	return ret;
    }
    
    private boolean tankNummerExclusive(String c,String nummer) {
    	boolean ret = !(
    			c.contains(nummer +"/") ||
				c.contains("/" + nummer) ||
				c.contains(nummer + ".") ||
    			c.contains("." + nummer) ||
    			c.contains(nummer +"-") ||
    			c.contains("-" + nummer)
    	);
    	return ret;
    }
    
    private void getMaxCommentSizeOnce() {
    	try {
    		Statement stm = con.createStatement();
    		ResultSet rs = stm.executeQuery("SELECT character_maximum_length " + 
        			"FROM information_schema.columns " + 
        			"WHERE column_name = 'comment' " + 
        			"AND table_name = 'buchungen' ");
    		if(rs.first()) {
    			this.max_comment_size=rs.getInt("character_maximum_length");
    		}
    	}catch (final Exception e){
			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchungen.getMaxCommentSize" , e.getLocalizedMessage());
		}
    }
    public int getMaxCommentSize() {
    	return this.max_comment_size;
    }
    
    public void getZugangAbgangListe(ErgebnisListe liste){
    	 HashMap<Integer,letzteVerwendungItem> abgangsliste = new  HashMap<Integer,letzteVerwendungItem>();
    	 HashMap<Integer,letzteVerwendungItem> zugangsliste = new  HashMap<Integer,letzteVerwendungItem>();
    	 if(liste !=null && !liste.isEmpty()) {
	    	 try {
	    		 String ids="";
	    		 for(int i=0;i<liste.size();i++) {
	    			 ids += String.valueOf(liste.getItem(i).getID()) + ",";
	    		 }
	    		 if (ids.endsWith(",")){
	    			 ids=ids.substring(0, ids.length()-1);
	    		 }
	    		 if(ids.length()>0) {
	    			 Statement stmt = con.createStatement();
	    			 ResultSet rs = stmt.executeQuery("select rezeptur_id,max(datum) as datum1 from buchungen where typ=" + TYP_ABGANG + " and rezeptur_id in(" + ids + ") group by rezeptur_id;");
	    			 while(rs.next()) {
	    				 int id = rs.getInt("rezeptur_id");
	    				 Date datum = rs.getString("datum1")!=null?db.getDateFromString(rs.getString("datum1")):null;
    					 abgangsliste.put(id, new letzteVerwendungItem(id,datum));
	    			 }
	    			 rs = stmt.executeQuery("select rezeptur_id,max(datum) as datum1 from buchungen where typ=" + TYP_ZUGANG + " and rezeptur_id in(" + ids + ") group by rezeptur_id;");
	    			 while(rs.next()) {
	    				 int id = rs.getInt("rezeptur_id");
	    				 Date datum = rs.getString("datum1")!=null?db.getDateFromString(rs.getString("datum1")):null;
    					 zugangsliste.put(id, new letzteVerwendungItem(id,datum));
	    			 }
		    		 for(int i=0;i<liste.size();i++) {
		    			 int id = liste.getItem(i).getID();
		    			 if(abgangsliste.get(id) != null) liste.getItem(i).setAbgang(abgangsliste.get(id).getDatum());
	    				 if(zugangsliste.get(id) != null) liste.getItem(i).setZugang(zugangsliste.get(id).getDatum());
		    		 }
	    		 }
			}  	catch (final Exception e){
				rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getZugangAbgangListe" , e.getLocalizedMessage());
			}
    	 }
    }
    
    public double getRestMenge(String losnummer,int einheit) {
    	double  ret = 0;
    	try {
    		getRestMenge.setString(1, losnummer);
    		final ResultSet rs = getRestMenge.executeQuery();
    		if(rs.next()) {
    			switch(einheit) {
    				case RzPro.EINHEIT_LITER:   ret = rs.getDouble("rest_liter");break;
    				case RzPro.EINHEIT_KG:   ret = rs.getDouble("rest_kg");break;
    				case RzPro.EINHEIT_LA:   ret = rs.getDouble("rest_la");break;
    			}
    		}
    	}  	catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getRestLiter" , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public String getLosnummerFromHerstellung(int id) {
    	String ret = "";
    	try {
    		getLosnummerFromHerstellung.setInt(1, id);
    		final ResultSet rs = getLosnummerFromHerstellung.executeQuery();
    		if(rs.next()) {
    			ret = rs.getString("losnummer");
    		}
    	}  	catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getLosnummerFromHerstellung" , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public ArrayList<BuchungLosnummerItem> getLosnummerSuche(String value){
    	ArrayList<BuchungLosnummerItem> ret  = new ArrayList<BuchungLosnummerItem>();
    	try {
    		if(!value.trim().equals("")) {
	    		getLosnummerSuchListe.setString(1, "^"+value);
	    		final ResultSet rs =getLosnummerSuchListe.executeQuery();
	    		while(rs.next()) {
	    			ret.add(new BuchungLosnummerItem(
	    					rz,
	    					rs.getInt("id"),
	    					rs.getInt("rezeptur_id"),
	    					rs.getString("bezeichnung"),
	    					rs.getString("losnummer"),
	     	                rs.getDouble("liter"),
	     	                db.getDateFromString(rs.getString("datum"))
	    					));
	    		}
    		}
    	}    	catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getLosnummerSuche" , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public String[] getZugangsBuchungByLosnummer(String value) {
    	String[] ret= {"",""};
    	try {
    		getZugangsBuchungByLosnummer.setString(1, value);
    		final ResultSet rs =getZugangsBuchungByLosnummer.executeQuery();
    		if(rs.next()) {
    			ret[0] = rs.getString("bezeichnung");
    			ret[1] = String.valueOf(rs.getInt("herstellung"));
    		}
    	}    	catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getLosnummerByID" , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public String getMhdStringHtml(long time) {
    	String ret="";
    	if(time !=0) {
    		mhd.setTimeInMillis(time);
    		if(time != 0) {
				String c1="";
				String c2="";
				if(mhd.before(now)) {
					c1="<font bgcolor=\"red\" color=\"white\"><b>";
					c2="</b></font>";
				}
				ret +="&nbsp;" + c1+rz.getLocale().getString("commentlosnummer.mhd").replaceAll("%s",monate[mhd.get(Calendar.MONTH)]+" " +mhd.get(Calendar.YEAR))+c2;
			}
    	}
    	return ret;
    }
    
    public String getMhdString(long time) {
    	String ret="";
    	if(time !=0) {
    		mhd.setTimeInMillis(time);
    		if(time != 0) {
    			ret += monate[mhd.get(Calendar.MONTH)]+" " +mhd.get(Calendar.YEAR);
				if(mhd.before(now)) {
					ret +=" " + rz.getLocale().getString("dbbuchung.mhd_abgelaufen");
				}
			}
    	}
    	return ret;
    }
    
    
    public ArrayList<mhdItem> getMhdListe(boolean sort,boolean reverse,Calendar von,Calendar bis){
    	//sort true=mhd,false=rezeptur
    	ArrayList<mhdItem> liste=new ArrayList<mhdItem>();
    	try {
    		PreparedStatement stm=null;
    		if(sort==true) {
    			if(reverse==true) {
    				stm = getMhdListeMhdReverseDatum;
    				getMhdListeMhdReverseDatum.setLong(1, von.getTimeInMillis());
    				getMhdListeMhdReverseDatum.setLong(2, bis.getTimeInMillis());
    			}else {
    				stm= getMhdListeMhdDatum;
    				getMhdListeMhdDatum.setLong(1, von.getTimeInMillis());
    				getMhdListeMhdDatum.setLong(2, bis.getTimeInMillis());
       			}
    		}else {
				if(reverse==true) {
					stm=getMhdListeRezepturReverseDatum;
    				getMhdListeRezepturReverseDatum.setLong(1, von.getTimeInMillis());
    				getMhdListeRezepturReverseDatum.setLong(2, bis.getTimeInMillis());
				}else {
					stm=getMhdListeRezepturDatum;
    				getMhdListeRezepturDatum.setLong(1, von.getTimeInMillis());
    				getMhdListeRezepturDatum.setLong(2, bis.getTimeInMillis());
				}
			}
    		final ResultSet rs=stm.executeQuery();
    		while (rs.next()) {
    			liste.add(new mhdItem(
    					rs.getInt("id"),
    					rs.getInt("rezeptur_id"),
    					rs.getString("bezeichnung"),
    					rs.getString("losnummer"),
    					rs.getLong("mhd"),
    					rs.getString("artikelnummer"),
    					rs.getInt("tempvon"),
    					rs.getInt("tempbis")
    					));
    			
    		}
	}
   	catch (final Exception e){
   		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getMhdListeDatum" , e.getLocalizedMessage());
   	}
    	return liste;
    }
    
    public ArrayList<mhdItem> getMhdListe(boolean sort,boolean reverse){
    	//sort true=mhd,false=rezeptur
    	ArrayList<mhdItem> liste=new ArrayList<mhdItem>();
    	try {
    		PreparedStatement stm=null;
    		if(sort==true) {
    			if(reverse==true) {
    				stm = getMhdListeMhdReverse;
    			}else {
    				stm= getMhdListeMhd;
       			}
    		}else {
				if(reverse==true) {
					stm=getMhdListeRezepturReverse;
				}else {
					stm=getMhdListeRezeptur;
				}
			}
    		final ResultSet rs=stm.executeQuery();
    		while (rs.next()) {
    			liste.add(new mhdItem(
    					rs.getInt("id"),
    					rs.getInt("rezeptur_id"),
    					rs.getString("bezeichnung"),
    					rs.getString("losnummer"),
    					rs.getLong("mhd"),
    					rs.getString("artikelnummer"),
    					rs.getInt("tempvon"),
    					rs.getInt("tempbis")
    					));
    			
    		}
	}
   	catch (final Exception e){
   		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getMhdListe" , e.getLocalizedMessage());
   	}
    	return liste;
    }
    
    public void saveMhd(int id,long datum) {
    	try {
    			saveMhd.setLong(1,datum);
    			saveMhd.setInt(2,id);
    			saveMhd.executeUpdate();
		}
	   	catch (final Exception e){
	   		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.saveMhd" , e.getLocalizedMessage());
	   	}
    }
    
    
    public String[] getMHdDaten(int id) {
    	String	ret[]= {"","","",""};
    	try {
			getMhdDaten.setInt(1,id);
			final ResultSet rs = getMhdDaten.executeQuery();
			if(rs.next()) {
				ret[0]=rs.getString("bezeichnung");
				ret[1]=rs.getString("losnummer");
				ret[2]=Long.toString(rs.getLong("mhd"));
				ret[3]=String.valueOf(rs.getInt("rezeptur_id"));
    		}    			
		}
	   	catch (final Exception e){
	   		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getMhdDaten" , e.getLocalizedMessage());
	   	}
    	return ret;
    }
    
    
    public Date getMinDatum(int rezeptur_id) {
    	Date	ret=null;
    	try {
			getErsteBuchung.setInt(1,rezeptur_id);
			final ResultSet rs = getErsteBuchung.executeQuery();
    		if(rs.next()) {
    			ret=db.getDateFromString(rs.getString("minDatum"));
    		}    			
		}
	   	catch (final Exception e){
	   		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getMinDatum" , e.getLocalizedMessage());
	   	}
    	return ret;
    }
    
    
   public ArrayList<ClientBuchungsItem> getBuchungenForToday(){
	   ArrayList<ClientBuchungsItem> ret= new ArrayList<ClientBuchungsItem>();
	   try {
        	final Date now = db.getServerTimestamp();
			SimpleDateFormat df=new SimpleDateFormat("yyyy-MM-dd 00:00:00");
			getBuchungenForToday.setString(1,db.dbGetUser());
			getBuchungenForToday.setString(2, df.format(now));
			final ResultSet rs = getBuchungenForToday.executeQuery();
    		while(rs.next()) {
    			ret.add(new ClientBuchungsItem(
    				rs.getInt("id"),
 	                db.getDateFromString(rs.getString("datum")),
 	                rs.getString("user"),
 	                rs.getString("bezeichnung"),
 	                rs.getDouble("liter"),
 	                rs.getString("losnummer")
    			));
    			
    		}
	}
   	catch (final Exception e){
   		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungenForToday" , e.getLocalizedMessage());
   	}
	return ret;	
   }
    
    
    public boolean isRProbe(int herstellung_id) {
    	boolean ret=false;
    	try {
    		if(herstellung_id != 0) {
	    		getRProbeErforderlich.setInt(1,herstellung_id);
	    		final ResultSet rs = getRProbeErforderlich.executeQuery();
	    		if(rs.next()) {
	    			ret=rs.getInt("r_probe")==0?false:true;
	    		}
	    	}
    	}
       	catch (final Exception e){
       		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.isRProbe" , e.getLocalizedMessage());
       	}
    	return ret;	
    }
    
    public int isRProbeVorhanden(String losnummer) {
    	int ret=0;
    	try {
    		getRProbeVorhanden.setString(1,losnummer);
    		final ResultSet rs = getRProbeVorhanden.executeQuery();
    		if(rs.next()) {
    			ret= rs.getInt("anzahl");
    		}
    	}
       	catch (final Exception e){
       		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.isRProbeVorhanden" , e.getLocalizedMessage());
       	}
    	return ret;	
    }
    /**
     * Tanks mit Bestand suchen
     * @param id
     * @return
     */
    public commentLosnummerListe getBestandTanks(int id,String tank,String sb,String se) {
    	commentLosnummerListe cll=new commentLosnummerListe(rz);
    	commentLosnummerItem cli = null;
    	try {
    		getBestandTanks.setInt(1,id);
    		final ResultSet rs = getBestandTanks.executeQuery();
    		while (rs.next()) {
    			long d = rs.getLong("mhd");
    			if(d == 0) {
    				cli = new commentLosnummerItem(rz,rs.getInt("buchungen.id"),rs.getString("buchungen.bezeichnung"),
    						rs.getDate("datum"),rs.getString("buchungen.losnummer"),rs.getString("comment"),
    						rz.getLocale().formatNumber(rs.getDouble("liter"),OptionFactory.NF_ONE) + " " + rz.getLocale().getString("string_liter_rest") ,tank,sb,se,rs.getInt("mischungen.id"));
    				cll.addItem(cli);
    			} else {
    				cli = new commentLosnummerItem(rz,rs.getInt("buchungen.id"),rs.getString("buchungen.bezeichnung"),
    						rs.getDate("datum"),rs.getString("buchungen.losnummer"),rs.getString("comment"),
    						rz.getLocale().formatNumber(rs.getDouble("liter"),OptionFactory.NF_ONE) + " " + rz.getLocale().getString("string_liter_rest") ,tank,sb,se,rs.getLong("mhd"),rs.getInt("mischungen.id"));
    			    cll.addItem(cli);	
    			}
    		}
    	}
       	catch (final Exception e){
       		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBestandTanks" , e.getLocalizedMessage());
       	}
    	return cll;
    }
     
    
     
    
    public int[] getHerstellungsRezepturen(int id) {
    	int[] ids = null;
    	ArrayList al = new ArrayList();
    	try {
    		getHerstellungsRezepturen.setInt(1,id);
    		final ResultSet rs=getHerstellungsRezepturen.executeQuery();
    		while (rs.next()){
    			al.add(Integer.valueOf(rs.getInt("rezeptur_id")));
    		}
    		if(!al.isEmpty()) {
    			ids = new int[al.size()];
    			for(int i = 0;i<al.size();i++) {
    				ids[i]=((Integer)al.get(i)).intValue();
    			}
    		}
    	}
       	catch (final Exception e){
       		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getVerbrauchListe" , e.getLocalizedMessage());
       	}
    	return ids;
    }
    
    /**
    *
    */
   public VerbrauchsListe getVerbrauch(ArrayList rezepturen) {
   	final VerbrauchsListe vl = new VerbrauchsListe();
   	try {
   		if( rezepturen.size() > 0) {
	   		Statement stm;
	   		ResultSet rs;
	   		String sql;
	   		String liste="";
	   		for(int i =0;i < rezepturen.size();i++) {
	   			final String v = String.valueOf((Integer)rezepturen.get(i));
	   			liste += v + ",";
	   			vl.getRezepturListe().add(rz.getDatabase().getRezeptur().dbGetRezepturName( Integer.parseInt(v)) );
	   		}
	   		if(liste.endsWith(",")) {
	   			liste = liste.substring(0,liste.length()-1);
	   		}
	        stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	        sql = "SELECT sum(liter) as sliter,sum(kg) as skg, sum(la) as sla, year(datum) as jahr FROM " + DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id in (" + liste + ") and typ=" +String.valueOf(TYP_ZUGANG)+  "  group by year(datum) order by year(datum) desc LIMIT " + rz.getOptionFactory().getOption("verbrauch.jahre",DEFAULT_VERBRAUCH_JAHRE);
	        rs=stm.executeQuery(sql);
	   		while(rs.next()) {
	   			vl.addZugang(new VerbrauchsItem(
	   					rs.getInt("jahr"),
	   					rs.getDouble("sliter"),
	   					rs.getDouble("skg"),
	   					rs.getDouble("sla")
	   			));
	   		}
	   		rs.close();
	        stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	        sql = "SELECT sum(liter) as sliter,sum(kg) as skg, sum(la) as sla, year(datum) as jahr FROM " + DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id in (" + liste + ") and typ=" +String.valueOf(TYP_ABGANG)+  "  group by year(datum) order by year(datum) desc LIMIT " +rz.getOptionFactory().getOption("verbrauch.jahre",DEFAULT_VERBRAUCH_JAHRE);
	        rs=stm.executeQuery(sql);
	   		while(rs.next()) {
	   			vl.addAbgang(new VerbrauchsItem(
	   					rs.getInt("jahr"),
	   					rs.getDouble("sliter"),
	   					rs.getDouble("skg"),
	   					rs.getDouble("sla")
	   			));
	   		}
	   		rs.close();
   		}
   	}
   	catch (final Exception e){
       		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getVerbrauchListe" , e.getLocalizedMessage());
   	}
   	return vl;
   }


    /**
     *
     */
    public VerbrauchsListe getVerbrauch(int rezeptur) {
    	Statement stm;
    	String sql;
    	final VerbrauchsListe vl = new VerbrauchsListe();
    	try {
        	stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    		ResultSet rs;
    		sql = "SELECT sum(liter) as sliter,sum(kg) as skg, sum(la) as sla, year(datum) as jahr FROM " + DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id=" + String.valueOf(rezeptur) +"  and typ=" +String.valueOf(TYP_ZUGANG)+  "  group by year(datum) order by year(datum) desc LIMIT " + rz.getOptionFactory().getOption("verbrauch.jahre",DEFAULT_VERBRAUCH_JAHRE);
    		rs=stm.executeQuery(sql);
    		while(rs.next()) {
    			vl.addZugang(new VerbrauchsItem(
    					rs.getInt("jahr"),
    					rs.getDouble("sliter"),
    					rs.getDouble("skg"),
    					rs.getDouble("sla")
    			));
    		}
    		rs.close();
    		sql = "SELECT sum(liter) as sliter,sum(kg) as skg, sum(la) as sla, year(datum) as jahr FROM " + DBFactory.TABLE_BUCHUNG + " WHERE rezeptur_id=" + String.valueOf(rezeptur) + " and typ=" + String.valueOf(TYP_ABGANG) + "  group by year(datum) order by year(datum) desc LIMIT " +  rz.getOptionFactory().getOption("verbrauch.jahre",DEFAULT_VERBRAUCH_JAHRE);
    		rs=stm.executeQuery(sql);
    		while(rs.next()) {
    			vl.addAbgang(new VerbrauchsItem(
    					rs.getInt("jahr"),
    					rs.getDouble("sliter"),
    					rs.getDouble("skg"),
    					rs.getDouble("sla")
    			));
    		}
    		rs.close();
    		vl.setBezeichnung(rz.getDatabase().getRezeptur().dbGetRezepturName(rezeptur));
    	}
    	catch (final Exception e){
        		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getVerbrauch" , e.getLocalizedMessage());
    	}
    	return vl;
    }

    /**
     * holt die Bemerkungen zur Wareneingangsbuchung  einer bestimmten Losnummer
     */
    public String getComment(String losnummer) {
    	String ret = "";
    	try {
    		if((losnummer != null) && (!losnummer.equals(""))){
    			getEinzelComment.setInt(1,TYP_ZUGANG);
    			getEinzelComment.setString(2,losnummer.trim());
    			final ResultSet rs = getEinzelComment.executeQuery();
    			if (rs.next()) {
    				ret = rs.getString(1);
    			}
    		}
    	}
    	catch (final Exception e){
        		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.setComment " , e.getLocalizedMessage());
        		ret = "";
    	}
    	return ret;

    }

    /**
     * speichert nur die Bemerkungen zu einer Buchung
     *
     */
    public boolean setComment(String comment,int id) {
    	boolean ret = true;
    	try {
    		if (comment==null) {
				comment = "";
			}
    		setEinzelComment.setString(1,comment);
    		setEinzelComment.setInt(2,id);
    		setEinzelComment.executeUpdate();
    	}
    	catch (final Exception e){
        		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.setComment " , e.getLocalizedMessage());
        		ret = false;
    	}
    	return ret;
    }

    /**
     * gibt den Typ einer Buchung (Zu/Abgang) als String zurück;
     * @param typ TYP_ZUGANG oder TYP_ABGANG
     * @return String
     */
    public String getTypToString(int typ) {
        switch (typ) {
	        case 		TYP_ZUGANG 				: return rz.getLocale().getString("database.buchung.typ_zugang");
	        case 		TYP_ABGANG					: return rz.getLocale().getString("database.buchung.typ_abgang");
        }
        return "";
    }

    /**
     * gibt den Code einer Buchung als STring zurück
     * @param code  CODE_HERSTELLUNG,CODE_EINKAUF usw....
     * @return   string
     */
    public String getCodeToString(int code) {
        switch (code) {
	        case 		CODE_HERSTELLUNG 				: return rz.getLocale().getString("database.buchung.code_herstellung");
	        case 		CODE_EINKAUF  					: return rz.getLocale().getString("database.buchung.code_einkauf");
	        case 		CODE_ENTNAHME 					: return rz.getLocale().getString("database.buchung.code_entnahme");
	        case 		CODE_SCHWUND					: return rz.getLocale().getString("database.buchung.code_schwund");
	        case 		CODE_SCHWUNDAUSGLEICH 	: return rz.getLocale().getString("database.buchung.code_schwundausgleich");
	        case 		CODE_KORREKTUR_ZUGANG	: return rz.getLocale().getString("database.buchung.code_korrektur_zugang");
	        case 		CODE_KORREKTUR_ABGANG  : return rz.getLocale().getString("database.buchung.code_korrektur_abgang");
        }
        return "";
    }

    
    private BuchungsItem getBuchungFromRSOhneMischung(ResultSet rs,boolean rezeptur_holen) {
    	BuchungsItem ret = null;
    	try {
    		ret = new BuchungsItem(
       			 	rz,
	                rs.getInt("buchungen.id"),
	                rs.getInt("typ"),
	                rs.getInt("code"),
					rs.getInt("lager"),
	                rs.getInt("herstellung"),
	                rs.getInt("buchungen.rezeptur_id"),
	                null, //Rezeptur brauchen wir hier nicht.
	                db.getDateFromString(rs.getString("datum")),
	                rs.getString("buchungen.user"),
	                rs.getString("buchungen.bezeichnung"),
	                rs.getString("buchungen.losnummer"),
	                rs.getInt("losnummer_reihe"),
	                rs.getString("comment"),
	                rs.getBoolean("erledigt"),
	                rs.getBoolean("uebernommen"),
	                rs.getInt("einheit"),
	                rs.getDouble("staerke"),
	                rs.getDouble("litergewicht"),
	                rs.getDouble("liter"),
	                rs.getDouble("kg"),
	                rs.getDouble("buchungen.la"),
	                rs.getDouble("rest_liter"),
	                rs.getDouble("rest_kg"),
	                rs.getDouble("rest_la"),
	                rs.getBoolean("rest"),
	                rezeptur_holen, // Nein, wir holen die Rezeptur NICHT !!!!!
	                rs.getLong("mhd"),
	               0,
	               rs.getString("steuer_artikel"),
	               rs.getInt("sorte"),
	               rs.getInt("steuerlager"),
	               rs.getString("sorten.bezeichnung"),
	               rs.getString("steuer_lager.bezeichnung")
       		);
    	}  catch (final Exception e){
        	rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungFromRS", e.getLocalizedMessage());
        	ret = null;
    	}
    	return ret;
    }
    
    private BuchungsItem getBuchungFromRS(ResultSet rs,boolean rezeptur_holen) {
    	BuchungsItem ret = null;
    	try {
    		ret = new BuchungsItem(
       			 	rz,
	                rs.getInt("buchungen.id"),
	                rs.getInt("typ"),
	                rs.getInt("code"),
					rs.getInt("lager"),
	                rs.getInt("herstellung"),
	                rs.getInt("buchungen.rezeptur_id"),
	                null, //Rezeptur brauchen wir hier nicht.
	                db.getDateFromString(rs.getString("datum")),
	                rs.getString("buchungen.user"),
	                rs.getString("buchungen.bezeichnung"),
	                rs.getString("buchungen.losnummer"),
	                rs.getInt("losnummer_reihe"),
	                rs.getString("comment"),
	                rs.getBoolean("erledigt"),
	                rs.getBoolean("uebernommen"),
	                rs.getInt("einheit"),
	                rs.getDouble("staerke"),
	                rs.getDouble("litergewicht"),
	                rs.getDouble("liter"),
	                rs.getDouble("kg"),
	                rs.getDouble("buchungen.la"),
	                rs.getDouble("rest_liter"),
	                rs.getDouble("rest_kg"),
	                rs.getDouble("rest_la"),
	                rs.getBoolean("rest"),
	                rezeptur_holen, // Nein, wir holen die Rezeptur NICHT !!!!!
	                rs.getLong("mhd"),
	                rs.getInt("mischungen.id"),
		            rs.getString("steuer_artikel"),
		            rs.getInt("sorte"),
		            rs.getInt("steuerlager"),
		            rs.getString("sorten.bezeichnung"),
		            rs.getString("steuer_lager.bezeichnung")
       		);
    	}  catch (final Exception e){
        	rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungFromRS", e.getLocalizedMessage());
        	ret = null;
    	}
    	return ret;
    }
    
    /**
     * gibt die Liste aller Buchungen zu einer losnummer zurück
     * @param bi   BuchungsItem
     */
    public StatistikListenItem getStatistikListe(BuchungsItem bi) {
    	StatistikListenItem ret = new StatistikListenItem(bi);
		try {
            ResultSet rs = null;
            	getStatistikListe.setString(1,bi.getLosnummer());
                rs = getStatistikListe.executeQuery();
	            while(rs.next()) {
            	ret.getListe().addItem(getBuchungFromRS(rs,false));
            }
            if(ret.getListe().size() == 0) {
            	ret = null;
            } else {
	            for(int i=0;i < ret.getListe().size();i++) {
	            	if(ret.getListe().getItem(i).getHerstellung() != 0) {
						ret.getListe().getItem(i).setHerstellung_losnummer(getHerstellungsLosnummer(ret.getListe().getItem(i)));
					}
	            }
	            final int rezeptur = ret.getListe().getItem(0).getRezeptur_id();
	            final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	            final String sql = "SELECT liter,kg,la,rest_liter,rest_kg,rest_la FROM "+ DBFactory.TABLE_BUCHUNG +
	            	" WHERE rezeptur_id="+String.valueOf(rezeptur) + " and typ = 1 and rest=1";
	            rs = stm.executeQuery(sql);
	            while(rs.next()) {
	            	ret.setGesamtLiter(ret.getGesamtLiter() + rs.getDouble("liter"));
	            	ret.setGesamtKg(ret.getGesamtKg() + rs.getDouble("kg"));
	            	ret.setGesamtLa(ret.getGesamtLa() + rs.getDouble("la"));
	            	ret.setGesamtRestLiter(ret.getGesamtRestLiter() + rs.getDouble("rest_liter"));
	            	ret.setGesamtRestKg(ret.getGesamtRestKg() + rs.getDouble("rest_kg"));
	            	ret.setGesamtRestLa(ret.getGesamtRestLa() + rs.getDouble("rest_la"));
	            }
            }
    	}  catch (final Exception e){
        	rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getStatistikListe", e.getLocalizedMessage());
        	ret = null;
    	}
    	return ret;
    }

    /** gibt alle Buchungen zu einer Rezeptur-ID zurück, deren
     * Rest ungleich 0 ist (==Bestand), und zwar für eine Liste
     * von Rezepturen
     */
    public BuchungsListe getBestandsListeGruppe(ErgebnisListe liste,SaldenItem si) {
    	BuchungsListe ret = new BuchungsListe();
		try {
			ResultSet rs;
			String rezepturen="";
			for(int i = 0;i < liste.size();i++) {
				final ErgebnisItem ei = ((ErgebnisItem)liste.getItem(i));
				rezepturen += ei.getID() + ",";
			}
			if(rezepturen.endsWith(",")) {
				rezepturen = rezepturen.substring(0,rezepturen.length()-1);
			}
			if(!rezepturen.equals("")) {
				final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
				String sql = "SELECT * FROM "+ DBFactory.TABLE_BUCHUNG +
					" left join rezeptliste on buchungen.rezeptur_id=rezeptliste.id " + 
            		" left join sorten on buchungen.sorte=sorten.id  " + 
            		" left join steuer_lager on buchungen.steuerlager=steuer_lager.id " +
            		" WHERE rezeptur_id in ("+ rezepturen + ") and buchungen.typ=1 and rest=1 ORDER BY datum";
				rs = stm.executeQuery(sql);
				while (rs.next()) {
					Date d=new Date();
					d.setTime(0);
					ret.add(getBuchungFromRSOhneMischung(rs,false));
				}
				sql = "SELECT sum(rest_liter)as summe_liter,sum(rest_kg) as summe_kg, sum(rest_la) as summe_la FROM "+ DBFactory.TABLE_BUCHUNG +
        			" WHERE rezeptur_id in ("+ rezepturen + ") and typ = 1 and rest=1";
				rs = stm.executeQuery(sql);
				while(rs.next()) {
	                si.setLiter(rs.getDouble("summe_liter"));
	                si.setKg(rs.getDouble("summe_kg"));
	                si.setLa(rs.getDouble("summe_la"));
				}
			}
    	}  catch (final Exception e){
        	rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBestandsListeGruppe", e.getLocalizedMessage());
        	ret = null;
    	}
		if(ret != null) {
	        for(int i=0;i < ret.size();i++) {
	        	if(ret.getItem(i).getHerstellung() != 0) {
					//ret.getItem(i).setHerstellung_losnummer(getHerstellungsLosnummer(ret.getItem(i)));
					setHerstellungsDaten(ret.getItem(i));
				}
	        }
		}
    	return ret;
    }


    /** gibt alle Buchungen zu einer Rezeptur-ID zurück, deren
     * Rest ungleich 0 ist (==Bestand)
     */
    public BuchungsListe getBestandsListe(int rezeptur,SaldenItem si) {
    	BuchungsListe ret = new BuchungsListe();
    	if(rezeptur != 0) {
    		try {
    			ResultSet rs;
	            final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	            String sql = "SELECT * FROM "+ DBFactory.TABLE_BUCHUNG +
	            		" left join rezeptliste on rezeptliste.id = buchungen.rezeptur_id "+
	            		" left join sorten on buchungen.sorte=sorten.id  " + 
	            		" left join steuer_lager on buchungen.steuerlager=steuer_lager.id " +
	            	" WHERE rezeptur_id="+String.valueOf(rezeptur) + " and buchungen.typ=1 and rest=1";
	            rs = stm.executeQuery(sql);
    			while (rs.next()) {
    				Date d=new Date();
    				d.setTime(0);
    				ret.add(getBuchungFromRSOhneMischung(rs,false));

    			}
				sql = "SELECT sum(rest_liter)as summe_liter,sum(rest_kg) as summe_kg, sum(rest_la) as summe_la FROM "+ DBFactory.TABLE_BUCHUNG +
    			" WHERE rezeptur_id="+ rezeptur + " and typ=1 and rest=1";
				rs = stm.executeQuery(sql);
				while(rs.next()) {
	                si.setLiter(rs.getDouble("summe_liter"));
	                si.setKg(rs.getDouble("summe_kg"));
	                si.setLa(rs.getDouble("summe_la"));
				}
    			if(rs != null) {
    				rs.close();
    			}
        	}  catch (final Exception e){
            	rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBestandsListe", e.getLocalizedMessage());
            	ret = null;
        	}
    	}
    	if(ret != null) {
	        for(int i=0;i < ret.size();i++) {
	        	if(ret.getItem(i).getHerstellung() != 0) {
					//ret.getItem(i).setHerstellung_losnummer(getHerstellungsLosnummer(ret.getItem(i)));
					setHerstellungsDaten(ret.getItem(i));
				}
	        }
    	}
    	return ret;
    }

    /**
     * gibt alle Buchungen zu einer bestimmten Losnummer aus
     * @param string mit dem Muster zur Losnummer
     * @return Buchungsliste
     */
    public BuchungsListe getLosnummerHistory(String l,String b,String nummer) {
    	BuchungsListe ret = new BuchungsListe();
    	if(true) {
    		l =  l.replaceAll("\\s+",".*").replaceAll("[()\\[\\]{}\'\"]","");
    		b =  b.replaceAll("\\s+",".*").replaceAll("[()\\[\\]{}\'\"]","");
    		nummer =  nummer.replaceAll("\\s+",".*").replaceAll("[()\\[\\]{}\'\"]","");    		    		
    		if(!l.equals(".*")) {
    	        try {
    	            final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    	        	//String sql = "SELECT distinct buchungen.id," + BUCHUNG_FIELD_LIST + " FROM " + DBFactory.TABLE_BUCHUNG + 
    	    	    String sql = "SELECT  * FROM " + DBFactory.TABLE_BUCHUNG + 
    	        			" left join rezeptliste on rezeptliste.id = buchungen.rezeptur_id left join mischungen on buchungen.losnummer=mischungen.losnummer " +
    	            		" left join sorten on buchungen.sorte=sorten.id  " + 
    	            		" left join steuer_lager on buchungen.steuerlager=steuer_lager.id " +
    	        			"  WHERE ";
    	        	String where = "";
    	        	if(!l.equals("")) {
    	        		where += "buchungen.losnummer regexp '" + l + "' ";
    	        	}
    	        	if(!b.equals("")) {
    	        		where += "AND buchungen.comment regexp '" + b + "' ";
    	        	}
    	        	if(nummer != null && !nummer.equals("")) {
    	        		where += getBuchungsNummerSQL(nummer);
    	        	}
    	        	if(where.startsWith("AND")) {
    	        		where = where.substring(4);
    	        	}
    	        	if(where.startsWith("OR")) {
    	        		where = where.substring(3);
    	        	}
    	        	if (stm != null) {
	    	            final ResultSet rs = stm.executeQuery(sql + where + " ORDER by buchungen.id ASC");
	    	            while (rs.next()) {
	    	            	Date d=new Date();
	    	            	d.setTime(0);
	    	            	ret.add(getBuchungFromRS(rs,false));
	
	    	            }
	    	            if(rs != null){
	    	                rs.close();
	    	            }
	    	            /* die Losnummer der Herstellung wird nicht mit einem JOIN ermittelt, weil das nicht immer
	    	             * notwendig ist
	    	             */
	    	            for(int i=0;i < ret.size();i++) {
	    	            	if(ret.getItem(i).getHerstellung() != 0) {
								//ret.getItem(i).setHerstellung_losnummer(getHerstellungsLosnummer(ret.getItem(i)));
	    	            		setHerstellungsDaten(ret.getItem(i));
							}
	    	            }
    	        	}
    	        }  catch (final Exception e){
    	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungsListe", e.getLocalizedMessage());
    	            ret = null;
    	        }
    		}
    	}
    	return ret;
    }
    
    private void setHerstellungsDaten(BuchungsItem bi) {
        try {
            getHerstellungsDaten.setInt(1,bi.getHerstellung());
            final ResultSet rs = getHerstellungsDaten.executeQuery();
            while(rs.next()) {
            	bi.setHerstellung_losnummer(rs.getString("herstellungen.losnummer"));
            	bi.setHerstellung_bezeichnung(rs.getString("herstellungen.bezeichnung"));
            	bi.addHerstellungProbe(rs.getInt("proben.id"));
            }
            if(rs != null) {
                rs.close();
            }
        }	catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.setHerstellungsDaten", e.getLocalizedMessage());
        }
    }
    
    
    /**
     * prüft, ob eine bestimmte Rezeptur schon einmal verbucht wurde
     * @param id  die id der REzeptur
     * @return true, wenn verwendet
     */
    public boolean checkRezepturUsed(int id) {
        boolean ret = true;
        try {
            checkRezepturUsed.setInt(1, id);
            final ResultSet rs = checkRezepturUsed.executeQuery();
            if(rs.next()) {
                ret = true;
            } else {
                ret = false;
            }
            if(rs != null) {
                rs.close();
            }
        }	catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.checkRezepturUsed", e.getLocalizedMessage());
            ret = true;
        }
        return ret;
    }

    /**
     * setzt Zahlen auf null, die eine gewisse Abweichung von 0 haben
     * verhindert so, dass bei Abgängen von Losnummern immer 0 am Schluß
     * übrigbleibt und nicht etwa 0.00000000012312312309394 ;))
     * @param in
     * @return 0, wenn in <= precision oder in, wenn in > precision
     */
    private double setPrecision(double in) {
        if(Math.abs(in) < precision) {
            return 0.0;
        }
		return in;
    }

    public String getSteuerLagerSumme(LagerListenItem ll) {
    	String ret = "";
    	try {
    		if(ll.getListe().size() >0) {
	        	ResultSet rs;
	        	String herstellungen = "";
	        	for(int i = 0;i < ll.getListe().size();i++) {
	        		herstellungen += String.valueOf(((Integer)ll.getListe().get(i)).intValue()) + ",";
	        	}
	        	if(herstellungen.endsWith(",")) {
					herstellungen = herstellungen.substring(0, herstellungen.length()-1);
				}
	            final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	            final String sql = "SELECT steuer_artikel,lager,sum(liter) as sliter,sum(kg) as skg,sum(la) as sla,lager.* FROM " + DBFactory.TABLE_BUCHUNG +
	            	" right join lager on buchungen.lager=lager.id WHERE herstellung in(" + herstellungen + ") group BY steuer_artikel";
	            rs = stm.executeQuery(sql);

	            while(rs.next()) {
	            	if(!rs.getString("steuer_artikel").equals("")) {
		            	ret +=  rz.getLocale().getString("dbbuchung.label_saldo") +
		            			rs.getString("steuer_artikel") + " : " +
			                	"     " + rz.getLocale().formatNumber(rs.getDouble("sla"), OptionFactory.NF_NORMAL) +
			                	"  " + rz.getLocale().getString("string_la") + "\n"
			                	;
			            }
		            }
		            if(rs != null) {
		                rs.close();
		            }	
	            }
    	}catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getSteuerLagerSumme", e.getLocalizedMessage()+":"+e.getErrorCode());
            ret = "";
        }
    	return ret;
    }

    /**
     * gibt den fuss einer Lagerliste aus, d.h. die Salden der Buchungen einer Lagerliste,
     * summiert über einzelne Sorten
     * @return String  (mehrzeilig) zur Ausgabe in der Lagerliste
     */
    public String[] getLagerSumme(LagerListenItem ll) {
        String ret[] =  {"",""};
        String bezeichnung = "";
        if(ll.getListe().size() > 0) {
	        try {

	        	ResultSet rs;
	        	String herstellungen = "";
	        	for(int i = 0;i < ll.getListe().size();i++) {
	        		herstellungen += String.valueOf(((Integer)ll.getListe().get(i)).intValue()) + ",";
	        	}
	        	if(herstellungen.endsWith(",")) {
					herstellungen = herstellungen.substring(0, herstellungen.length()-1);
				}
	            final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
	            final String sql = "SELECT bezeichnung,steuer_artikel,lager,sum(liter) as sliter,sum(kg) as skg,sum(la) as sla,lager.* FROM " + DBFactory.TABLE_BUCHUNG +
	            	" right join lager on buchungen.lager=lager.id WHERE herstellung in(" + herstellungen + ") group BY bezeichnung";
	            rs = stm.executeQuery(sql);


	            while(rs.next()) {
	            	ret[0] += " -" + rs.getString("lager.name") + "- " +
	                	rs.getString("bezeichnung")  +  ( rs.getString("steuer_artikel").equals("")?"": " ("+rs.getString("steuer_artikel")+") " ) +
	                	"     " + rz.getLocale().formatNumber(rs.getDouble("sla"), OptionFactory.NF_NORMAL) +
	                	"  " + rz.getLocale().getString("string_la") + "\n";
	                if(!rs.getString("lager.ersatztext").equals("")) {
	                	bezeichnung = rs.getString("lager.ersatztext") +
	                			 ( rs.getString("steuer_artikel").equals("")?"": " ("+rs.getString("steuer_artikel")+") " )
	                			;
	                }else {
	                	bezeichnung = rs.getString("bezeichnung") +
	                			 ( rs.getString("steuer_artikel").equals("")?"": " ("+rs.getString("steuer_artikel")+") " )
	                			 ;
	                }
	                if(!rz.isZero(rs.getDouble("sla"))) {
	                	ret[1] += " -" + rs.getString("lager.name") + "- "+
	                	bezeichnung+
	                	"     " + rz.getLocale().formatNumber(rs.getDouble("sla"), OptionFactory.NF_NORMAL) +
	                	"  " + rz.getLocale().getString("string_la") + "\n";
	                }
	            }
	            if(rs != null) {
	                rs.close();
	            }
	            final String sql2 = "SELECT sum(la) as sla1,sum(endmenge_la) as sla  FROM " + DBFactory.TABLE_HERSTELLUNG+
		            	"  WHERE id in(" + herstellungen + ") ";
		        rs = stm.executeQuery(sql2);
            	String summe = "\n\n";
	            if(rs.next()) {
	            	summe += rz.getLocale().getString("report.lagerliste.zugang") + rz.getLocale().formatNumber(rs.getDouble("sla1"),OptionFactory.NF_NORMAL) + " " + rz.getLocale().getString("string_la") + "\n";
	            	summe += rz.getLocale().getString("report.lagerliste.zugang_schwund") + rz.getLocale().formatNumber(rs.getDouble("sla1") - rs.getDouble("sla"),OptionFactory.NF_NORMAL) + " " + rz.getLocale().getString("string_la") + "\n";
	            	summe += rz.getLocale().getString("report.lagerliste.zugang_endmenge") + rz.getLocale().formatNumber(rs.getDouble("sla"),OptionFactory.NF_NORMAL) + " " + rz.getLocale().getString("string_la") + "\n";
	            }
	            ret[0] += summe += "\n\n"+getSteuerLagerSumme(ll);
	            ret[1] += summe;
	        } catch(final SQLException e) {
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getLagerSumme", e.getLocalizedMessage()+":"+e.getErrorCode());
	            ret = new String[2];
	        }
        }
        return ret;
    }

    /**
     * druckt die Buchungsliste, getrennt nach Lagern aus
     * @param mode
     * @param von
     * @param bis
     * @param bezeichnung
     * @param losnummer
     */
    public LagerDruckListe buchungsListeDrucken(int mode ,Date von, Date bis,String bezeichnung,String invers,String losnummer,String user,LagerListe lager) {
        ResultSet rs;
        Statement stm;
        String sql;
        LagerDruckListe ld = null;
        try {
            // temporäre Tabelle mit allen sichtbaren Buchungen anlegen
            stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            sql=" SELECT " + DBFactory.TABLE_BUCHUNG + ".herstellung," +
            	DBFactory.TABLE_BUCHUNG + ".id ," + DBFactory.TABLE_BUCHUNG +".datum, lager.liste FROM " + DBFactory.TABLE_BUCHUNG +
            	" INNER JOIN " + DBFactory.TABLE_LAGER+ " ON " + DBFactory.TABLE_BUCHUNG + ".lager=" + DBFactory.TABLE_LAGER +".id ";
            final String where = getHerstellungenWhereClause(true,mode,von,bis,bezeichnung,invers,losnummer,user,lager);
            if(!where.equals("")) {
                sql += " WHERE " +where + " AND ";
            } else {
                sql += " WHERE ";
            }
            sql +=  "  typ=1 AND code=1 AND herstellung <> 0";
            rs = stm.executeQuery(sql);
            ld=db.getLager().getLagerListenNummern();
            ld.setMode(mode);
            ld.setBezeichnung(bezeichnung);
            ld.setVon(von);
            ld.setBis(bis);
            ld.setLosnummer(losnummer);
            while (rs.next()) {
                ld.addItemToListe(rs.getInt("liste"),rs.getInt("herstellung"));
            }
            if(rs != null) {
                rs.close();
            }
            if(stm != null) {
                stm.close();
            }
        }catch( final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.buchungsListeDrucken", e.getLocalizedMessage());
            ld=null;
        }
        return ld;
    }

    /**
     * 
     * liefert eine Herstellung zu einer losnummer
     */
    public HerstellungsItem getHerstellungFromLosnummer(String losnummer) {
    		HerstellungsItem ret = null;
    		try {
    		getHerstellungFromLosnummer.setString(1, losnummer);
    		final ResultSet rs = getHerstellungFromLosnummer.executeQuery();
    		if(rs.next()) {
    			ret = getEinzelHerstellung(rs.getInt("id"));
    		}
    		} catch(final SQLException e) {
    			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getHerstellungFromLosnummer", e.getLocalizedMessage());
    		}
    return ret;
    	
    }
    /**l
     * liefert die Liste aller Buchungen einer bestimmten herstellung
     * @param rs
     * @return
     */
    public BuchungsListe getHerstellung(int herstellung) {
        final BuchungsListe bl = new BuchungsListe();
        try {
            getHerstellung.setInt(1,herstellung);
            final ResultSet rs = getHerstellung.executeQuery();
            while (rs.next()) {
            	BuchungsItem bi =getEinzelBuchung(rs.getInt("id"));
                bl.add(bi);
            }
            if(rs != null) {
                rs.close();
            }
        } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getHerstellung", e.getLocalizedMessage());
        }
        return bl;
    }

    public BuchungsItem getHerstellungZugang(int herstellung) {
    	final BuchungsItem ret = null;
    	BuchungsListe bl = getHerstellung(herstellung);
    	if(bl.size()>0) {
    		for(int i = 0;i < bl.size();i++) {
    			if(bl.getItem(i).getTyp() == DBBuchung.TYP_ZUGANG ) {
    				return bl.getItem(i);
    			}
    		}
    	}
    	return ret;
    }
    
    /**
     * holt die Losnummer der Herstellung zu einer Buchung,
     * wenn die Buchung einer Herstellung angehört
     * @param buchungsItem
     * @return null, wenn die Buchung keiner Herstellung zugehört, STRING wenn ja.
     */
    public String getHerstellungsLosnummer(BuchungsItem bi) {
    	String ret = null;
        try {
        	if(bi.getHerstellung() != 0) {
	            getHerstellungsLos.setInt(1,bi.getHerstellung());
	            final ResultSet rs = getHerstellungsLos.executeQuery();
	            while (rs.next()) {
	            	ret = rs.getString("herstellungen.losnummer");
	            }
	            if(rs != null) {
	                rs.close();
	            }
        	}
        } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getHerstellungsLosnummer", e.getLocalizedMessage());
        }
    	return ret;
    }
    /**
     * holt eine einzelne Herstellung nach ihrer ID aus der Datenbank
     * @param id
     * @return
     */
    public HerstellungsItem getEinzelHerstellung(int id) {
        try {
            getEinzelHerstellung.setInt(1,id);
            final ResultSet rs = getEinzelHerstellung.executeQuery();
            if(rs.next()) {
            	return new HerstellungsItem(rz,
		                rs.getInt("id"),
		                rs.getInt("rezeptur"),
		                rs.getString("bezeichnung"),
		                db.getDateFromString(rs.getString("datum")),
		                rs.getString("losnummer"),
		                rs.getBoolean("erledigt"),
		                rs.getBoolean("uebernommen"),
		                rs.getDouble("liter"),
		                rs.getDouble("kg"),
		                rs.getDouble("la"),
		                rs.getDouble("schwundsatz"),
		                rs.getDouble("endmenge_liter"),
		                rs.getDouble("endmenge_kg"),
		                rs.getDouble("endmenge_la"),
		                rs.getString("user"),
		                0,"","","","",0,rs.getInt("r_probe")==0?false:true,
		                false
		                );
            }
            
        } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getEinzelherstellung.resultset", e.getLocalizedMessage());
            return null;
        }
        return null;
    }

    /**
     * holt eine einzelne Herstellung aus der Datenbank
     * @param rs   Resultset, aus dem die Herstellung entnommen wird
     * @return
     */
    public HerstellungsItem getEinzelHerstellung(ResultSet rs) {
        try {
        return new HerstellungsItem(rz,
                rs.getInt("id"),
                rs.getInt("rezeptur"),
                rs.getString("bezeichnung"),
                db.getDateFromString(rs.getString("datum")),
                rs.getString("losnummer"),
                rs.getBoolean("erledigt"),
                rs.getBoolean("uebernommen"),
                rs.getDouble("liter"),
                rs.getDouble("kg"),
                rs.getDouble("la"),
                rs.getDouble("schwundsatz"),
                rs.getDouble("endmenge_liter"),
                rs.getDouble("endmenge_kg"),
                rs.getDouble("endmenge_la"),
                rs.getString("user") ,
                rs.getInt("buchungen.lager"),
                rs.getString("lager.hg"),
                rs.getString("lager.vg"),
                rs.getString("tank"),
                rs.getString("proben.losnummer"),
                rs.getInt("proben.id"),
                rs.getInt("r_probe") ==0?false:true,
                rs.getInt("buchungen.rest")==0?false:true
                );

        } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getEinzelherstellung.resultset", e.getLocalizedMessage());
            return null;
        }
    }


    
    /**
     * liefert den SQL-String Where-Clause im Herstellbuchfenster
     * @param ausstattung
     * @param mode
     * @param von
     * @param bis
     * @param bezeichnung
     * @param losnummer
     * @return
     */
	private String getHerstellungenWhereClause(boolean buchungen,int mode ,Date von, Date bis,String bezeichnung,String invers,String losnummer,String user,LagerListe lager) {
        String ret = "";
        switch(mode) {
    	case HerstellungsFenster.AUSWAHL_ALLE :
    		break;
    	case HerstellungsFenster.AUSWAHL_ERLEDIGT :
    	    ret += " AND " + (buchungen?" erledigt ":" herstellungen.erledigt ") + " <> 0 ";
    		break;
    	case HerstellungsFenster.AUSWAHL_UNERLEDIGT :
    	    ret += " AND " + (buchungen?" erledigt ":" herstellungen.erledigt ") + " = 0 ";
    		break;
        }
        if((von != null) && (bis != null)) {
        	SimpleDateFormat df=new SimpleDateFormat("yyyy-MM-dd");
            ret += " AND (" + (buchungen?" datum ": " herstellungen.datum") + " BETWEEN '" + df.format(von) + " 00:00:00' AND '" + df.format(bis) + " 23:59:59') ";
        }
        if(bezeichnung != null) {
            if(!bezeichnung.equals("")) {
				ret += " AND " + (buchungen?" bezeichnung ":" herstellungen.bezeichnung ") + " REGEXP '" + bezeichnung.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(invers != null) {
            if(!invers.equals("")) {
				ret += " AND " + (buchungen?" bezeichnung ":" herstellungen.bezeichnung ") +" NOT REGEXP '" + invers.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(losnummer != null) {
            if(!losnummer.equals("")) {
				ret += " AND " + (buchungen?" losnummer ":" herstellungen.losnummer ") + " REGEXP '" + losnummer.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(user != null) {
            if(!user.equals("")) {
				ret += " AND " + (buchungen?" user ":" herstellungen.user ") + " REGEXP '" + user.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(lager != null) {
        	String insert = "";
        	if(lager.getChecked(false)) {
        		// wenn KEINE Lager ausgewählt sind wird einfach lager==1 gesucht,
        		// was zum Ergebnis hat, dass auch nichts mehr gefunden wird.
        		insert += "lager=-1";
        	} else {
	        	for(int i = 0;i < lager.size();i++) {
	        		if(lager.getItem(i).isChecked()){
	        			insert += "buchungen.lager=" + String.valueOf(lager.getItem(i).getID()) + " OR ";
	        		}
	        	}
        	}
        	if(insert.endsWith(" OR ")) {
				insert=insert.substring(0,insert.length()-" OR ".length());
			}
        	if(!insert.equals("")) {
				ret += " AND ("+insert + ") ";
			}
        }
        ret=ret.trim();
        if(ret.startsWith("AND")) {
			ret = ret.substring("AND".length());
		}
        return ret.trim();
    }

   /**
    * Statsitik-Auswertung nach Rezeptur
    * gibt eine Liste der Herstellungen nach Rezeptur zurück
    */
    public StatistikRezepturListe getStatistikRezeptur(String Sortierung,StatistikRezepturListe sli) {
        Statement stm;
        String sql;
        try {
        	final int mode = sli.getMode();
        	final Date von = sli.getVon();
        	final Date bis = sli.getBis();
        	final String bezeichnung = sli.getBezeichnung();
        	final String invers = sli.getInvers();
        	final String losnummer=sli.getLosnummer();
        	final LagerListe lager = sli.getLager();
        	sli.clear();
        	String user="";
            stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            final String whereClause = getHerstellungenWhereClause(false,mode,von,bis,bezeichnung,invers,losnummer,user,lager) ;
            sql = "SELECT sum(herstellungen.liter) as sl, sum(herstellungen.la) as sla,herstellungen.*,buchungen.typ,buchungen.code,buchungen.lager,lager.hg,lager.vg from herstellungen " +
             		" left join buchungen on buchungen.herstellung=herstellungen.id left join lager on lager.id=buchungen.lager "  +
             		" where buchungen.typ=1 and buchungen.code=1 " +
             		(whereClause.equals("")?"":"  AND " + whereClause) +
             		 "GROUP BY bezeichnung  ORDER BY " + Sortierung;
            final ResultSet rs = stm.executeQuery(sql);
            while (rs.next()) {
                sli.addItem(new StatistikRezepturItem(rz,rs.getString("bezeichnung"),rs.getDouble("sl"),rs.getDouble("sla")));
            	}
        	} catch(final Exception e) {
                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getStatistik", e.getLocalizedMessage());
        	}
        	return sli;
        }

    /**
     * liefert eine Liste der Herstellungen;
     * @param mode
     * @param von
     * @param bis
     * @param bezeichnung
     * @param losnummer
     *
     * @return
     */
    public HerstellungsListe getHerstellungen(int mode ,Date von, Date bis,String bezeichnung,String invers,String losnummer,String user,LagerListe lager,SaldenItem herstellmenge,SaldenItem schwund,SaldenItem endmenge) {
        final HerstellungsListe hl=new HerstellungsListe();
        Statement stm;
        String sql;
        try {
            stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            final String whereClause = getHerstellungenWhereClause(false,mode,von,bis,bezeichnung,invers,losnummer,user,lager) ;
            sql = "SELECT herstellungen.*,buchungen.rest,buchungen.typ,buchungen.code,buchungen.lager,lager.hg,lager.vg,proben.losnummer,proben.id from herstellungen " +
             		" left join buchungen on buchungen.herstellung=herstellungen.id left join lager on lager.id=buchungen.lager "  +
             		"left join proben on proben.losnummer=herstellungen.losnummer "+
             		" where buchungen.typ=1 and buchungen.code=1 " +
             		(whereClause.equals("")?"":"  AND " + whereClause) +
             		 "  ORDER BY herstellungen.datum";
            ResultSet rs = stm.executeQuery(sql);
            while (rs.next()) {
            		hl.add(getEinzelHerstellung(rs));
            }
            if(rs != null){
                rs.close();
            }
            if(stm != null) {
                stm.close();
                stm=null;
            }
            stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            sql = "SELECT sum(herstellungen.liter) as m_liter,sum(herstellungen.kg) as m_kg,sum(herstellungen.la) as m_la," +
            	"sum(herstellungen.endmenge_liter) as e_liter,sum(herstellungen.endmenge_kg) as e_kg,sum(herstellungen.endmenge_la) as e_la FROM " +
            	DBFactory.TABLE_HERSTELLUNG + " " +
            	" left join buchungen on buchungen.herstellung=herstellungen.id left join lager on lager.id=buchungen.lager "+
         		" where buchungen.typ=1 and buchungen.code=1 " +
         		(whereClause.equals("")?"":" AND  " + whereClause) ;
            	rs = stm.executeQuery(sql);
            if(rs.next()) {
                herstellmenge.setLiter(rs.getDouble("m_liter"));
                herstellmenge.setKg(rs.getDouble("m_kg"));
                herstellmenge.setLa(rs.getDouble("m_la"));
                endmenge.setLiter(rs.getDouble("e_liter"));
                endmenge.setKg(rs.getDouble("e_kg"));
                endmenge.setLa(rs.getDouble("e_la"));
                schwund.setLiter(Math.abs(herstellmenge.getLiter()-endmenge.getLiter()));
                schwund.setKg(Math.abs(herstellmenge.getKg()-endmenge.getKg()));
                schwund.setLa(Math.abs(herstellmenge.getLa()-endmenge.getLa()));
            }
            if(rs != null){
                rs.close();
            }
            if(stm != null) {
                stm.close();
            }

        }  catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getHerstellungsListe", e.getLocalizedMessage());
        }
        return hl;
    }


    /**
     * gibt die Liste der Herstellungen, aufbereitet für die
     * Monatsstatistik, zurück
     * Hier werden die Summen für jeden Monat ermittelt und als
     * WerteListe zurückgegeben
     *
     */
    public WerteListe getHerstellungsStatistik(int mode ,Date von, Date bis,String bezeichnung,String invers,String losnummer,String user,LagerListe lager) {
    	final WerteListe wi = new WerteListe();
    	Statement stm;
    	String sql;
        try {
            stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            final String where =getHerstellungenWhereClause(false,mode,von,bis,bezeichnung,invers,losnummer,user,lager) ;
            sql = "SELECT month(herstellungen.datum) as monat,year(herstellungen.datum) as jahr,sum(herstellungen.liter) as sliter," +
            " sum(herstellungen.kg) as skg, sum(herstellungen.la) as sla, sum(herstellungen.endmenge_liter) as seliter," +
            " sum(herstellungen.endmenge_kg) as sekg,sum(herstellungen.endmenge_la) as sela from " + DBFactory.TABLE_HERSTELLUNG + " " +
            " left join buchungen on buchungen.herstellung=herstellungen.id left join lager on lager.id=buchungen.lager " +
            	(where.equals("")?"WHERE code=1 and typ=1 ":" WHERE typ=1 and code=1 and " + where ) +
            	" group by year(datum),month(datum)";
            final ResultSet rs = stm.executeQuery(sql);
            while (rs.next()) {
            	wi.addItem(new WerteItem(
            			rs.getInt("jahr"),
            			rs.getInt("monat"),
            			rs.getDouble("sliter"),
            			rs.getDouble("seliter"),
            			rs.getDouble("skg"),
            			rs.getDouble("sekg"),
            			rs.getDouble("sla"),
            			rs.getDouble("sela")
            	));
          	}
         }catch (final SQLException e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.getHerstellungsStatistik", e.getLocalizedMessage());
         }
    	return wi;
    }


    /** prüft, ob eine gegebene Losnummer schon existiert
     *
     * @param losnummer
     * @return true, wenn existent
     */
    public boolean checkLosnummer(String losnummer) {
        boolean ret = true;
        try{
            checkLosnummerExists.setString(1,losnummer.trim());
            final ResultSet rs = checkLosnummerExists.executeQuery();
            if(!rs.next()) {
                ret = false;
            }
       }
       catch (final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.checkLosnummer", e.getLocalizedMessage());
           ret = true;
       }
        return ret;
    }

    /** liefert einen SQL-String um die Salden zu ermitteln
     *
     * @param rezeptur
     * @param mode
     * @param von
     * @param bis
     * @param benutzer
     * @param losnummer
     * @return
     */
    private String getSaldenSQL(int rezeptur, int mode ,Date von, Date bis,String benutzer,String losnummer,String bemerkungen,String buchungsnummer,boolean korrektur) {
        String ret = "SELECT sum(liter) as summe_liter,sum(kg) as summe_kg,sum(la) as summe_la FROM " + DBFactory.TABLE_BUCHUNG + " ";
        ret += getBuchungenWhereClause(String.valueOf(rezeptur),mode,von,bis,benutzer,losnummer,bemerkungen,buchungsnummer,korrektur);
        return ret;
    }

    /**
     * liefert den SQL-String Where-Clause für Buchungen mit der Auswahl von,bis,benutzer,losnummer
     * @param rezeptur
     * @param mode
     * @param von
     * @param bis
     * @param benutzer
     * @param losnummer
     * @return
     */
	private String getBuchungenWhereClause(String rezeptur, int mode ,Date von, Date bis,String benutzer,String losnummer,String bemerkungen,String buchungsnummer,boolean korrektur) {
        String ret = " WHERE buchungen.rezeptur_id " ;
        if(rezeptur.indexOf(",") != -1) {
        	ret +=" in(" + rezeptur + ") ";
        } else {
        	ret += "=" + rezeptur + " ";
        }
    	switch(mode) {
    	case StatistikFenster.AUSWAHL_ALLE :
    		break;
    	case StatistikFenster.AUSWAHL_ZUGANG :
    	    ret += " AND buchungen.typ=1 ";
    		break;
    	case StatistikFenster.AUSWAHL_ABGANG :
    	    ret += " AND  buchungen.typ=2 ";
    		break;
        }
        if((von != null) && (bis != null)) {
        	SimpleDateFormat df=new SimpleDateFormat("yyyy-MM-dd");
            ret += " AND ( buchungen.datum BETWEEN '" + df.format(von) + " 00:00:00' AND '" + df.format(bis) + " 23:59:59') ";
        }
        if(benutzer != null) {
            if(!benutzer.equals("")) {
				ret += " AND buchungen.user REGEXP '" + benutzer.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(losnummer != null) {
            if(!losnummer.equals("")) {
				ret += " AND buchungen.losnummer REGEXP '" + losnummer.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(bemerkungen != null) {
            if(!bemerkungen.equals("")) {
				ret += " AND buchungen.comment REGEXP '" + bemerkungen.replaceAll("\\s+",".*") + "' ";
			}
        }
        if(buchungsnummer != null) {
            if(!buchungsnummer.equals("")) {
            	ret += getBuchungsNummerSQL(buchungsnummer);
			}
        }
        if(korrektur) {
        	ret += " AND (buchungen.code = " + DBBuchung.CODE_KORREKTUR_ABGANG + " OR buchungen.code = " + DBBuchung.CODE_KORREKTUR_ZUGANG+" ) ";
        }
        return ret ;
    }

	private String getBuchungsNummerSQL(String buchungsnummer) {
		String ret = "";
		String single = "";
		String range = "";
		if(!buchungsnummer.contains("|"))buchungsnummer = buchungsnummer+"|"; 
		String[] liste = buchungsnummer.split("\\|");
    	for(int i = 0;i<liste.length;i++) {
    		if(liste[i].contains("-")){
    			range +="OR " + splitItem(liste[i]);
    		} else {
    			single += liste[i] + ",";
    		}
    	}
		if(!single.trim().isEmpty()) {
			if(single.endsWith(","))single = single.substring(0,single.length()-1);
			ret += "OR  buchungen.id in(" + single.trim() +")";
		}
		if(!range.trim().isEmpty()) {
			ret +=range.trim() + " ";
		}
		return ret ;
	}
	
//	private String getBuchungsNummerSQL1(String buchungsnummer) {
//		String ret  = "";
//		String  b = "";
//    	if(buchungsnummer.contains("|")) {
//    		String[] liste = buchungsnummer.split("\\|");
//    		for(int i = 0;i<liste.length;i++) {
//    			b += splitItem(liste[i]);
//    		}
//    	} else {
//    		b = splitItem(buchungsnummer);
//    	}
//    	if(b.endsWith("|"))b=b.substring(0, b.length()-1);
//    	if(!b.trim().equals("")) {
//    		b = b.replaceAll("\\s+"," ").replaceAll("\\|", ",");
//    		ret = "AND buchungen.id IN (" + b + ") ";
//    	}
//		return ret;
//	}
	
	private String splitItem(String in) {
		String out = "";

			String[] werte = in.split("-");
			if(werte.length >= 2) {
				try {
					int iOne = Integer.valueOf(werte[0]);
					int iTwo = Integer.valueOf(werte[1]);
					if(iOne >= iTwo) {
						int z = iOne;
						iOne = iTwo;
						iTwo = z;
					}				
					String s1 = Integer.toString(iOne);
					String s2 = Integer.toString(iTwo);
					if(s1.length() < s2.length()) {
						int l = s2.length()-s1.length();
						String s3 = s2.substring(0,l) + s1;
						iTwo = Integer.parseInt(s3);	
						iOne = Integer.valueOf(werte[0]);
					}
					out = " (buchungen.id BETWEEN " + iOne + " AND " + iTwo +") ";
				}catch (Exception e) {}
			}
		return out;
	}
	
    /**
     * liefert die Salden aller im Statistikfenster sichtbaren Buchungen
     * @param rezeptur
     * @param mode
     * @param von
     * @param bis
     * @param benutzer
     * @param losnummer   Auswahl
     * @return SaldenItem
     */
    public SaldenItem getBuchungenSaldo(int rezeptur,int mode,Date von, Date bis,String benutzer,String losnummer,String bemerkungen,String buchungsnummer,boolean korrektur) {
        SaldenItem ret = null;
        try {
            final String sql=getSaldenSQL(rezeptur,mode,von,bis,benutzer,losnummer,bemerkungen,buchungsnummer,korrektur);
            final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            final ResultSet rs=stm.executeQuery(sql);
            if (rs.next()) {
                ret = new SaldenItem(
                rs.getDouble("summe_liter"),
                rs.getDouble("summe_kg"),
                rs.getDouble("summe_la"),
                RzPro.EINHEIT_LITER
                );
            }
            if(rs != null) {
                rs.close();
            }
            if(stm != null) {
                stm.close();
            }
        }  catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungenSaldo", e.getLocalizedMessage());
            ret = null;
        }
        return ret;
    }

   
    
    
    /**
     * liefert eine Liste aller Buchungen nach der Auswahl im Statistikfenster für eine einzelne
     * Rezeptur
     * @param rezeptur   die Liste aller zur Rezeptur gehörenden Buchungen
     * @param mode
     * @param von
     * @param bis
     * @param benutzer
     * @param losnummer
     * @return
     */
    public BuchungsListe getBuchungen(int rezeptur,int mode ,Date von, Date bis,String benutzer,String losnummer,String bemerkungen,String buchungsnummer,boolean korrektur) {
        BuchungsListe ret = new BuchungsListe();
        try {
            final Statement stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            final String sql = "SELECT " + BUCHUNG_FIELD_LIST + "  FROM " + DBFactory.TABLE_BUCHUNG +
            			" left join rezeptliste on buchungen.rezeptur_id=rezeptliste.id left join mischungen on buchungen.losnummer=mischungen.losnummer left join sorten on buchungen.sorte=sorten.id left join steuer_lager on buchungen.steuerlager=steuer_lager.id " +
            			getBuchungenWhereClause(String.valueOf(rezeptur) ,mode,von,bis,benutzer,losnummer,bemerkungen,buchungsnummer,korrektur) + " ORDER BY datum";
            final ResultSet rs = stm.executeQuery(sql);
            while (rs.next()) {
            	ret.add( getBuchungFromRS(rs,false) );

            }
            if(rs != null){
                rs.close();
            }
            if(stm != null) {
                stm.close();
            }
            /* die Losnummer der Herstellung wird nicht mit einem JOIN ermittelt, weil das nicht immer
             * notwendig ist
             */
            for(int i=0;i < ret.size();i++) {
            	if(ret.getItem(i).getHerstellung() != 0) {
					//ret.getItem(i).setHerstellung_losnummer(getHerstellungsLosnummer(ret.getItem(i)));
            		setHerstellungsDaten(ret.getItem(i));
				}
            }
        }  catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungsListe", e.getLocalizedMessage());
            ret = null;
        }
        return ret;
    }

    /**
     * Liefert eine Liste aller Buchungen, die zu den in der @param liste aufgeführten REzepturen gehören.
     * @param liste
     * @param si
     * @param mode
     * @param von
     * @param bis
     * @param benutzer
     * @param losnummer
     * @return Buchungsliste, die alle Buchungen enthält
     * Ausserdem wird ein SaldenItem mit den entsprechenden Daten gefüllt
     */
    
    
    public BuchungsListe getBuchungen(ErgebnisListe liste,SaldenItem si,int mode ,Date von, Date bis,String benutzer,String losnummer,String bemerkungen,String buchungsnummer,boolean korrektur) {
        BuchungsListe ret = new BuchungsListe();
        try {
            ResultSet rs = null;
            int i;
            final Statement stm= con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            String rezeptur ="";
            for(i = 0;i < liste.size();i++) {
            	rezeptur += String.valueOf(((ErgebnisItem)liste.getItem(i)).getID()) + ",";
            }
            if(rezeptur.endsWith(",")) {
				rezeptur = rezeptur.substring(0,rezeptur.length()-1);
			}

            String sql = "SELECT " +" *" +"  FROM " + DBFactory.TABLE_BUCHUNG +
            	    " left join rezeptliste on buchungen.rezeptur_id = rezeptliste.id "+	
            		" left join sorten on buchungen.sorte=sorten.id  " + 
            		" left join steuer_lager on buchungen.steuerlager=steuer_lager.id " +
            		" " + getBuchungenWhereClause(rezeptur,mode,von,bis,benutzer,losnummer,bemerkungen,buchungsnummer,korrektur) + " ORDER BY datum";
            rs = stm.executeQuery(sql);
            while(rs.next()) {
            	ret.add( getBuchungFromRSOhneMischung(rs,false) );
            }
            sql = "SELECT sum(liter) as summe_liter,sum(kg) as summe_kg,sum(la) as summe_la FROM " +DBFactory.TABLE_BUCHUNG + " " + getBuchungenWhereClause(rezeptur,mode,von,bis,benutzer,losnummer,bemerkungen,buchungsnummer,korrektur);
            rs = stm.executeQuery(sql);
            while (rs.next()) {
                si.setLiter(rs.getDouble("summe_liter"));
                si.setKg(rs.getDouble("summe_kg"));
                si.setLa(rs.getDouble("summe_la"));
            }
            if(rs != null) {
                rs.close();
            }
            /* die Losnummer der Herstellung wird nicht mit einem JOIN ermittelt, weil das nicht immer
             * notwendig ist
             */
            for(i=0;i < ret.size();i++) {
            	if(ret.getItem(i).getHerstellung() != 0) {
					//ret.getItem(i).setHerstellung_losnummer(getHerstellungsLosnummer(ret.getItem(i)));
					setHerstellungsDaten(ret.getItem(i));
				}
            }
        }  catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBuchungsListeTemp", e.getLocalizedMessage());
            ret = null;
        }
        return ret;
    }

    /**
     * holt eine einzelne Buchung aus einem ResultSet
     */
    private BuchungsItem getBuchungFromResultSet(ResultSet rs) {
    	BuchungsItem ret = null;
    	try {
	    	 ret=  getBuchungFromRS(rs,true) ;
    	} catch (final Exception e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.dbGetBuchungFromResultSet", e.getLocalizedMessage());
    	}
    	return ret;
    }
  

    /**
     * holt eine einzelne Buchung;
     * @param aID
     * @return
     */
    public BuchungsItem getEinzelBuchung(int aID) {
        BuchungsItem ret = null;
        try{
            if(aID != 0) {
                getBuchung.setInt(1,aID);
                final ResultSet rs = getBuchung.executeQuery();
                if(rs.next()) {
                	ret = getBuchungFromResultSet(rs);
                }
                if(rs != null) {
                    rs.close();
                }
            }
       }
       catch (final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.dbGetBuchung", e.getLocalizedMessage());
           ret = null;
       }
       return ret;
    }

    /**
     * speichert eine einzelne Buchung
     * @param bi
     * @return
     */
    public boolean saveEinzelBuchung(BuchungsItem bi) {
        boolean ret = true;
        double staerke=0;
        double litergewicht=rz.getTafelFactory().Litergewicht(0);
        try{
        	staerke=bi.getStaerke();
        	litergewicht = bi.getKg()/bi.getLiter();
            saveBuchung.setInt(1,bi.getTyp());
            saveBuchung.setInt(2, bi.getCode());
            saveBuchung.setInt(3,bi.getLager());
            saveBuchung.setInt(4,bi.getHerstellung());
            saveBuchung.setInt(5,bi.getRezeptur_id());
            saveBuchung.setString(6,db.dbGetDateFormatString( bi.getDatum()));
            saveBuchung.setString(7,bi.getUser());
            saveBuchung.setString(8, bi.getBezeichnung());
            saveBuchung.setString(9,bi.getLosnummer());
            saveBuchung.setInt(10,bi.getLosnummer_reihe());
            saveBuchung.setString(11,bi.getComment());
            saveBuchung.setBoolean(12, bi.isErledigt());
            saveBuchung.setBoolean(13, bi.isUebernommen());
            saveBuchung.setInt(14,bi.getEinheit());
            saveBuchung.setDouble(15,staerke);
            saveBuchung.setDouble(16,litergewicht);
            saveBuchung.setDouble(17,bi.getLiter());
            saveBuchung.setDouble(18,bi.getKg());
            saveBuchung.setDouble(19,bi.getLa());
            saveBuchung.setDouble(20,bi.getRest_liter());
            saveBuchung.setDouble(21,bi.getRest_kg());
            saveBuchung.setDouble(22,bi.getRest_la());
            double v=0;
            switch(bi.getEinheit()) {
            	case RzPro.EINHEIT_KG : v=bi.getRest_kg();break;
            	case RzPro.EINHEIT_LITER : v = bi.getRest_liter();break;
            	case RzPro.EINHEIT_LA : v = bi.getRest_la();break;
            }
            if(!rz.isZero(v)) {
                saveBuchung.setBoolean(23,true);
            } else {
                saveBuchung.setBoolean(23,false);
            }
           	saveBuchung.setLong(24, bi.getMhdLong());
           	saveBuchung.setString(25, bi.getSteuer_artikel());
           	saveBuchung.setInt(26, bi.getSorte());
           	saveBuchung.setInt(27, bi.getSteuerlager());
           	
            final int r = saveBuchung.executeUpdate();
            if(r == 0) {
                ret = false;
            } else {
                bi.setId(db.dbGetLastInsertID(DBFactory.TABLE_BUCHUNG));
            }
       }
       catch (final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.saveEinzelBuchung", e.getLocalizedMessage());
           ret = false;
       }
       return ret;
    }

     /**
      * holt eine einzelne Buchung
      * @param aID die ID der Buchung (wenn 0, wird eine neue Buchung zurückgegeben
      * @return BuchungsItem
      */
     public BuchungsItem  dbGetBuchung(int aID,int rezeptur){
         BuchungsItem ret = null;
         try{
             if(aID == 0) {
            	 	SteuerDatenItem si = rz.getDatabase().getRezeptur().getSteuerDaten(rezeptur);
                 ret = new BuchungsItem(
                         rz,
                         0,
                         TYP_ZUGANG,
                         CODE_EINKAUF,
						 0,
                         0,
                         rezeptur,
                         null,//rezeptur == 0 ? null : db.getRezeptur().getRezepturBuchung(rezeptur),
                         db.getServerTimestamp(),
                         db.dbGetUser(),
                         "",
                         "",
                         REIHE_FIFO,
                         "",
                         false,
                         false,
                         0,
                         0,
                         0,

                         0,
                         0,
                         0,

                         0,
                         0,
                         0,
                         false,
                         true,
	     	             0,
	     	             0,
	     	             si.getSteuer_artikel(),
	     	             si.getSorte(),
	     	             si.getSteuerlager(),
	     	             si.getStrSorte(),
	     	             si.getStrSteuerlager()
                 ) ;
             } else {
                 ret = getEinzelBuchung(aID);
             }
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.dbGetBuchung", e.getLocalizedMessage());
            ret = null;
        }
        return ret;
     }

     /**
      * gibt eine Liste der Losnummern zurück, die bei einem Abgang betroffen wären
      * @param rezeptur
      * @param menge
      * @return
      */
     public LosnummerListe getLosNummerListe(int reihe,int rezeptur,double menge,EinheitsItem ei) {
         final LosnummerListe lose = new LosnummerListe(rz);
         String col="";
         switch(ei.getTyp()) {
         	case RzPro.EINHEIT_LITER:col = "rest_liter";break;
         	case RzPro.EINHEIT_KG:col = "rest_kg";break;
         	case RzPro.EINHEIT_LA:col = "rest_la";break;
         }
         double rest = menge;
         double get=0;
         double liter =0;
         double kg = 0;
         double la = 0;
         try {
             PreparedStatement getLosnummerListe;
             if(reihe == REIHE_FIFO) {
                 getLosnummerListe = getLosnummerListeFIFO;
             } else {
                 getLosnummerListe = getLosnummerListeLIFO;
             }
             if((rezeptur != 0) && !col.equals("")) {
                 getLosnummerListe.setInt(1,rezeptur);
                 final ResultSet rs = getLosnummerListe.executeQuery();
                 while (rs.next() && (rest > 0)) {
                     get = rs.getDouble(col);
                     if (rest > get) {
                         rest -= get;
                         liter = rz.getDatabase().getEinheiten().convert(get, rs.getDouble("staerke"), rs.getDouble("litergewicht"),ei, rz.getDatabase().getEinheiten().getEinheit("liter"));
                         kg = rz.getDatabase().getEinheiten().convert(get, rs.getDouble("staerke"), rs.getDouble("litergewicht"),ei, rz.getDatabase().getEinheiten().getEinheit("kg"));
                         la = rz.getDatabase().getEinheiten().convert(get, rs.getDouble("staerke"), rs.getDouble("litergewicht"),ei, rz.getDatabase().getEinheiten().getEinheit("la"));
                         lose.addItem(new LosnummerItem(
                                 rz,rs.getInt("id"),db.getDateFromString(rs.getString("datum")), rs.getInt("rezeptur_id"),rs.getString("losnummer"),rs.getString("comment"),rs.getInt("lager"),rs.getDouble("staerke"),liter,kg,la,rs.getString("bezeichnung"),rs.getInt("herstellung"),rs.getLong("mhd")
                         ));
                     }else {
                         liter = rz.getDatabase().getEinheiten().convert(rest, rs.getDouble("staerke"), rs.getDouble("litergewicht"),ei, rz.getDatabase().getEinheiten().getEinheit("liter"));
                         kg = rz.getDatabase().getEinheiten().convert(rest, rs.getDouble("staerke"), rs.getDouble("litergewicht"),ei, rz.getDatabase().getEinheiten().getEinheit("kg"));
                         la = rz.getDatabase().getEinheiten().convert(rest, rs.getDouble("staerke"), rs.getDouble("litergewicht"),ei, rz.getDatabase().getEinheiten().getEinheit("la"));
                         lose.addItem(new LosnummerItem(
                                 rz,rs.getInt("id"),db.getDateFromString(rs.getString("datum")), rs.getInt("rezeptur_id"),rs.getString("losnummer"),rs.getString("comment"),rs.getInt("lager"),rs.getDouble("staerke"),liter,kg,la,rs.getString("bezeichnung"),rs.getInt("herstellung"),rs.getLong("mhd")
                         ));
                         rest = 0;
                     }
                 }
                 if(rs != null) {
                     rs.close();
                 }
             }
         } catch (final SQLException e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getLosnummerListe", e.getLocalizedMessage());
         }
         return lose;
     }

     /** korrigiert eine einzelne Zugangsbuchung
      *  und speichert eine extra-AbgangsBuchung
      * @param li
      * @return
      */
     private boolean abgangLosnummer(LosnummerItem li,int herstellung,int code,String comment,Date datum) {
         boolean ret = false;
         BuchungsItem zugang = null;
         try {
             double liter=0;
             double kg = 0;
             double la = 0;
             if(li.getZugangsBuchung() != 0) {
                 zugang = getEinzelBuchung(li.getZugangsBuchung());
                 if((zugang != null) && (zugang.getId() != 0)) {
                     liter = setPrecision(zugang.getRest_liter() - li.getLiter());
                     if(rz.isZero(liter)) {
						liter = 0;
					}
                     kg = setPrecision(zugang.getRest_kg() - li.getKg());
                     if(rz.isZero(kg)) {
						kg=0;
					}
                     la = setPrecision(zugang.getRest_la() - li.getLa());
                     if(rz.isZero(la)) {
						la=0;
					}
                     zugangKorrektur.setDouble(1,liter);
                     zugangKorrektur.setDouble(2,kg);
                     zugangKorrektur.setDouble(3,la);
                     double v=0;
                     switch(zugang.getEinheit()) {
                     	case RzPro.EINHEIT_KG : v=kg;break;
                     	case RzPro.EINHEIT_LITER : v = liter;break;
                     	case RzPro.EINHEIT_LA : v = la;break;
                     }
                     if(!rz.isZero(v)) {
                         zugangKorrektur.setBoolean(4,true);
                     } else {
                         zugangKorrektur.setBoolean(4,false);
                     }
                     zugangKorrektur.setInt(5,li.getZugangsBuchung());
    	             if(zugangKorrektur.executeUpdate() != 0) {
    	                 final BuchungsItem bi = dbGetBuchung(0,li.getRezeptur());
                         bi.setDatum(datum);
    	                 if(bi != null) {
    	                     bi.setTyp(TYP_ABGANG);
    	                     bi.setCode(code);
    	                     bi.setHerstellung(herstellung);
    	                     bi.setLiter(-li.getLiter()); // entnahmen sind negativ
    	                     bi.setKg(-li.getKg());
    	                     bi.setLa(-li.getLa());
    	                     bi.setStaerke(li.getStaerke());
    	                     bi.setLosnummer(li.getLosnummer());
    	                     if(comment != null) {
    	                    	 bi.setComment(comment);
    	                     }
    	                     /*
    	                      * Ich weiss wirklich nicht mehr wofür die folgende
    	                      * Konstruktion mal gut war
    	                      * Geändert am 7.12.2009
    	                      */
//    	                     if((comment != null) && !comment.equals("")) {
//    	                         bi.setComment(comment);
//    	                     } else {
//    	                         bi.setComment(li.getComment());
//    	                     }
    	                     ret = saveEinzelBuchung(bi);
    	                 }
    	                 ret = true;
    	             }
                 }
             }
         } catch (final Exception e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".abgangBuchen", e.getLocalizedMessage());
             ret = false;
         }
         return ret;
     }

     /** bucht einen einzelnen Abgang mit evtl. mehreren Losnummern
      * MUSS in eine Transaktion gewrappt werden !
      * @param bi
      * @return
      */
     private boolean abgangBuchen(BuchungsItem bi) {
         boolean ret = true;
         try {
             // bestand ermitteln und buchen
             //double bestand = bi.getRezeptur().getBestand();
             double bestand = db.getRezeptur().getBestand( bi.getRezeptur_id());
             double value = 0;
             switch (bi.getRezeptur().getBestandseinheit()) {
             	case RzPro.EINHEIT_LITER: value = bi.getLiter(); break;
             	case RzPro.EINHEIT_KG: value = bi.getKg(); break;
             	case RzPro.EINHEIT_LA: value = bi.getLa(); break;
             }
             bestand +=value; // value ist negativ, daher vermindert das + den Bestand ;))
             if(db.getRezeptur().updateBestand(bi.getRezeptur_id(),bestand)){
                 /* wenn der Bestand unter 0 fällt, werden dafür keine Losnummern gefunden und
                  * auch nicht abgebucht... stattdessen werden 2 weitere Buchungen angelegt,
                  * eine Abgang/Schwund-Buchung und eine Zugang/Schwundausgleich-Buchung,
                  * weil der tatsächliche Bestand nie unter 0 fallen darf !
                 */
                 if(bestand < 0) {
                     if( Math.abs(bestand) >= precision) {
                         // wenn der Schwundbetrag größer als der Grenzwert ist, werden
                         // Korrekturbuchungen ausgeführt, ansonsten nicht !
	                     final double schwundValue = -(0-bestand);
		                 final BuchungsItem bi1 = dbGetBuchung(0,bi.getRezeptur_id());
	                     double liter = rz.getDatabase().getEinheiten().convert(schwundValue, bi.getStaerke(),bi.getLitergewicht(),rz.getDatabase().getEinheiten().getGrundeinheit(bi.getRezeptur().getBestandseinheit()), rz.getDatabase().getEinheiten().getEinheit("liter"));
	                     double kg = rz.getDatabase().getEinheiten().convert(schwundValue, bi.getStaerke(), bi.getLitergewicht(),rz.getDatabase().getEinheiten().getGrundeinheit(bi.getRezeptur().getBestandseinheit()), rz.getDatabase().getEinheiten().getEinheit("kg"));
	                     double la = rz.getDatabase().getEinheiten().convert(schwundValue, bi.getStaerke(), bi.getLitergewicht(),rz.getDatabase().getEinheiten().getGrundeinheit(bi.getRezeptur().getBestandseinheit()), rz.getDatabase().getEinheiten().getEinheit("la"));
	                     final String losnummer = this.getLosnummerVorschlag(bi.getRezeptur());
		                 if(bi1 != null) {
		                	 bi1.setDatum(bi.getDatum());
		                     bi1.setId(0);
		                     bi1.setTyp(TYP_ABGANG);
		                     bi1.setCode(CODE_SCHWUND);
		                     bi1.setHerstellung(bi.getHerstellung());
		                     bi1.setLager(bi.getLager());
		                     bi1.setLiter(liter);
		                     bi1.setKg(kg);
		                     bi1.setLa(la);
		                     bi1.setLosnummer(losnummer);
		                     bi1.setComment(rz.getLocale().getString("database.buchung.schwund_comment"));
		                     if(saveEinzelBuchung(bi1)){
		                         bi1.setId(0);
			                     bi1.setTyp(TYP_ZUGANG);
			                     bi1.setCode(CODE_SCHWUNDAUSGLEICH);
			                     bi1.setLiter(-liter); // Zahlen invertieren
			                     bi1.setKg(-kg);
			                     bi1.setLa(-la);
			                     bi1.setLosnummer(losnummer);
			                     bi1.setComment(rz.getLocale().getString("database.buchung.schwundausgleich_comment"));
			                     ret=saveEinzelBuchung(bi1);
		                     }
		                 }
                     } else {
                         //Der Wert, der unter 0 geht ist zu klein (< PRECISION) um bedeutsam zu
                         // sein. Daher wird der Bestand direkt auf 0 gesetzt und keine Schwundbuchungen eingefügt
                         // Der Bestand muss a1uch in der Rezeptur auf 0 gesetzt werden, weil die Korrekturen durch die
                         // oben angewandten Schwundbuchungen so nicht greifen.
                         bestand = 0;
                         db.getRezeptur().updateBestand(bi.getRezeptur_id(),bestand);
                     }

                 }
                 if(ret) {
	                 final LosnummerListe lose = getLosNummerListe(bi.getLosnummer_reihe(),bi.getRezeptur_id(),-value,rz.getDatabase().getEinheiten().getGrundeinheit(bi.getRezeptur().getBestandseinheit()));
	                 bi.setLosnummerListe(lose);
	                 for(int i =0;i < lose.size();i++) {
	                    if(! abgangLosnummer(lose.getItem(i),bi.getHerstellung(),bi.getCode(),bi.getComment(),bi.getDatum()))
	                    {
	                        ret=false;
	                        break;
	                    }
	                 }
                 }
             } else {
                ret = false;
             }
         } catch (final Exception e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".abgangBuchen", e.getLocalizedMessage());
             ret = false;
        }
     	BestandswarnungsItem bw = rz.getDatabase().getRezeptur().getBestandsWarnungEinzeln(bi.getRezeptur_id());
 	 	if(bw != null) {
 	 		if(!rz.getDatabase().getAuftrag().checkAuftragForRezeptur(bi.getRezeptur_id())) {
 	 			//Anfrage einstellen
 	 			AuftragsItem ai = new AuftragsItem(rz,bi.getRezeptur_id(),bi.getBezeichnung(),bw.getDiff());
 	 			if(ai != null) {
 	 				ai.setAnfrage(true);
 	 				try {
 	 	 				rz.getDatabase().getAuftrag().saveAuftrag(ai,rz.getFensterFactory().getFensterListe().getAuswahl());
 	 					if(rz.isMain()) { 	 						
 	 						Auswahl af = (Auswahl)rz.getFensterFactory().getFensterListe().getAuswahl();
 	 						af.getAnfragenListe();
 	 					} 
 	 				}catch(Exception e){
 	 					
 	 				}
 	 			}
 	 		}
 	 	}
         return ret;
     }
     /**
      * bucht einen einzelnen Zugang, MUSS in eine Transaktion gewrappt werden
      * @param bi
      * @return
      */
     private boolean zugangBuchen(BuchungsItem bi) {
         boolean ret = true;
         try {
             // bestand ermitteln und buchen
             //double bestand = bi.getRezeptur().getBestand();
             double bestand=db.getRezeptur().getBestand( bi.getRezeptur_id());
             switch (bi.getEinheit()) {
             	case RzPro.EINHEIT_LITER: bestand += bi.getLiter(); break;
             	case RzPro.EINHEIT_KG: bestand += bi.getKg(); break;
             	case RzPro.EINHEIT_LA: bestand += bi.getLa(); break;
             }
             if(db.getRezeptur().updateBestand(bi.getRezeptur_id(),bestand)){
               ret = saveEinzelBuchung(bi);
             } else {
                 ret = false;
             }
         } catch (final Exception e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".zugangBuchen", e.getLocalizedMessage());
             ret = false;
         }
         return ret;
     }

     /**
      * bucht einen einzelnen Zugang und wrappt den Vorgang in einer Transaction, die
      * die funktion zugangBuchen(bi) aufruft.
      * wird nur für Einzelbuchungen verwendet, Herstellungen, die mehrere Abgänge und
      * einen Zugang buchen, werden in herstellungBuchen() aufgerufen, da InnoDB
      * Transaktionen nicht verschachteln  kann
      * @param bi
      * @return
      */

     public int zugangBuchenTransaction(BuchungsItem bi) {
         int ret = 0;
         try {
        	 bi.setDatum(db.getServerTimestamp()); // Das Datum der Buchung setzen
             con.setAutoCommit(false);
             if(zugangBuchen(bi)) {
                 ret = bi.getId();
             } else {
                 ret = 0;
             }
             con.setAutoCommit(true);
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".zugangBuchenTransaction.rollback", ex.getLocalizedMessage());
	         ret = 0;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".zugangBuchenTransaction.2", excep.getLocalizedMessage());
			                ret = 0;
			            }
		            }
	         }
	     if (ret == 0) {
	         failMessage();
	     }
         return ret;
     }



     /**
      * bucht einen Abgang und legt dafür eine Transaction an
      * @param bi  BuchungsItem
      * @return true, wenn alles gebucht werden konnte, false, wenn abgebrochen werden musste.
      *
      */
     public boolean abgangBuchenTransaction(BuchungsItem bi) {
         boolean ret = true;
         try {
        	 bi.setDatum(db.getServerTimestamp()); // Das Datum der Buchung setzen
             con.setAutoCommit(false);
             if(!abgangBuchen(bi)) {
                 con.rollback();
                 ret = false;
             } else {
            	 	
             }
             con.setAutoCommit(true);
             rz.getDatabase();
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".abgangBuchenTransaction.rollback", ex.getLocalizedMessage());
	         ret = false;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".abgangBuchenTransaction.2", ex.getLocalizedMessage());
			                ret = false;
			            }
		            }
	         }
	     if (ret == false) {
	         failMessage();
	     }
         return ret;
     }

     public boolean mischungUebergangTransaction(BuchungsItem bi,int mischung) {
    	 boolean ret = true;
    	 try {
    		 bi.setDatum(db.getServerTimestamp()); // Das Datum der Buchung setzen
             con.setAutoCommit(false);
             if(abgangBuchen(bi)) {
            	 MischungsItem mi = rz.getDatabase().getMischungen().getMischung(mischung);
            	 LosnummerListe liste = bi.getLosnummerListe() ;
        		 if(mi != null) {
                	 for(int i=0;i<liste.size();i++) {
                		 LosnummerItem it=liste.getItem(i);
                		 TicketItem ti = new TicketItem(rz);
                		 if(ti != null) {
                			 ti.setMischung(mi.getHeader().getId());
                			 ti.setTyp(TicketItem.TYP_WARE);
                			 ti.setLa(it.getLa());
                			 ti.setStaerke(it.getStaerke());
                			 ti.setBezeichnung(it.getBezeichnung());
                			 ti.setBemerkungen(it.getComment());
                			 ti.setLosnummer_original(it.getLosnummer());
                			 ti.setRezeptur_id(it.getRezeptur());
                			 ti.setSorte(1);
                			 if(rz.getDatabase().getMischungen().saveEinzelTicket(mi, mi.getHeader().getTank(), ti)!= 0) {
                				 
                			 } else {
                				 ret = false;
                			 }
                		 } else {
                			 ret = false;
                		 }
                	 }	 
        		} else {
                	 ret = false;
                }
             }
            	if(!ret) {
            		con.rollback();
            	}
             con.setAutoCommit(true);
    	 }catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".mischungUbergangnTransaction.rollback", ex.getLocalizedMessage());
	         ret = false;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".mischungUebergangTransaction.2", ex.getLocalizedMessage());
			                ret = false;
			            }
		            }
	         }
    	 if (ret == false) {
		      failMessage();
		  }
    	 return ret;
     }
     
     public boolean umBuchenTransaction(BuchungsItem bi,int rezepturZugang) {
         boolean ret = true;
         if(rezepturZugang != 0 && rezepturZugang != bi.getRezeptur_id()) {
	         try {
	        	 bi.setDatum(db.getServerTimestamp()); // Das Datum der Buchung setzen
	             con.setAutoCommit(false);
	             if(abgangBuchen(bi)) {
	            	LosnummerListe liste = bi.getLosnummerListe() ;
	            	for(int i=0;i<liste.size();i++) {
	            		LosnummerItem it=liste.getItem(i);
			                final BuchungsItem bi1 = dbGetBuchung(0,rezepturZugang);
		                    bi1.setLiter(it.getLiter());
		                    bi1.setKg(it.getKg());
		                    bi1.setLa(it.getLa());
		                    bi1.setRest_la(it.getLa());
		                    bi1.setRest_kg(it.getKg());
		                    bi1.setRest_liter(it.getLiter());
			                bi1.setDatum(bi.getDatum());
		                    bi1.setId(0);
		                    bi1.setTyp(TYP_ZUGANG);
		                    bi1.setCode(CODE_EINKAUF);
		                    bi1.setHerstellung(bi.getHerstellung());
		                    bi1.setLager(bi.getLager());
		                    bi1.setLosnummer(it.getLosnummer());
		                    bi1.setComment(bi.getComment());
		                    if(!saveEinzelBuchung(bi1)) {
		                    	con.rollback();
		                    	ret = false;    
		                    	break;
		                    }
	            	}
	             }else {
	                 con.rollback();
	                 ret = false;            	 
	             }
	             con.setAutoCommit(true);
	             rz.getDatabase();
		     } catch(final Exception ex) {
		         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".abgangBuchenTransaction.rollback", ex.getLocalizedMessage());
		         ret = false;
		         if (con != null) {
			            try {
				            con.rollback();
				            con.setAutoCommit(true);
				            } catch(final Exception excep) {
				                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".abgangBuchenTransaction.2", ex.getLocalizedMessage());
				                ret = false;
				            }
			            }
		         }
		     if (ret == false) {
		         failMessage();
		     }
         }
         return ret;
     }

     public String getLosnummerVorschlagSpace(String in) {
    	 String ret = in;
    	 String temp = "";
    	 if(in.length()>1) {
    		 for(int i=1;i < in.length()-1;i++ ){
    			 temp = in.substring(0, i) + " " + in.substring(i);
    			 if(!checkLosnummer(temp)) {
    				 ret = temp;
    				 break;
    			 }
    		 }
    	 }
    	 return ret;
     }
     
     /**
      * holt das Template für den Losnummer-Vorschlag aus der Datenbank
      */
     public String getLosnummerVorschlagTemplate() {
    	  String ret = "";
          try {
              ResultSet rs;
              rs = getLosnummerVorschlagTemplate.executeQuery();
              if(rs.next()) {
              	ret = rs.getString("vtemplate").trim();
              }
              if(rs != null) {
                  rs.close();
              }
          } catch(final Exception e) {
  	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung:getLosnummerVorschlagTemplate", e.getLocalizedMessage());
  	         ret = "";
          }
          if(ret.equals("")) {
			ret = rz.getOptionFactory().getOption("losnummer.vtemplate");
		}
       	return ret;
     }



     /**
      * holt das Template für die Losnummer aus der Datenbank
      */
     public String getLosnummerTemplate() {
        String ret = "";
        try {
            ResultSet rs;
            rs = getLosnummerTemplate.executeQuery();
            if(rs.next()) {
            	ret = rs.getString("template").trim();
            }
            if(rs != null) {
                rs.close();
            }
        } catch(final Exception e) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung:getLosnummerTemplate", e.getLocalizedMessage());
	         ret = "";
        }
        if(ret.equals("")) {
			ret = OptionFactory.DEFAULT_LOSNUMMER_TEMPLATE;
		}
     	return ret;
     }

     /**
      * speichert das Template für die Losnummer in der Datenbank
      */
     public void setLosnummerTemplate(String template) {
        try {
        	setLosnummerTemplate.setString(1,template);
        	setLosnummerTemplate.executeUpdate();
        } catch(final Exception e) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung:setLosnummerTemplate", e.getLocalizedMessage());
        }
     }

     /**
      * speichert das Template für den Losnummer-vorschlag in der Datenbank
      */
     public void setLosnummerVorschlagTemplate(String template) {
        try {
        	setLosnummerVorschlagTemplate.setString(1,template);
        	setLosnummerVorschlagTemplate.executeUpdate();
        } catch(final Exception e) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung:setLosnummerVTemplate", e.getLocalizedMessage());
        }
     }
 /**
      * holt die nächste gültige Losnummer als String
      * @return losnummer
      */
     public String getLosnummerVorschlag(RezepturBuchungsItem ri) {
    	 return getLosnummerVorschlag1(ri.getTitel(),ri.getLosnummer(),ri.getArtikelnummer());
     }

     public String getLosnummerVorschlag(RezepturItem ri) {
    	 return getLosnummerVorschlag1(ri.getTitel(),ri.getLosnummer(),ri.getArtikelnummer());
     }

     public String getLosnummerVorschlag(MischungsItem mi,int rezeptur_id) {
    	 String ret = "";
    	 if(mi != null) {
    		int id = (rezeptur_id > 0)?rezeptur_id:mi.getHeader().getRezeptur_id();
    		if(id != 0) {
    			RezepturItem ri = rz.getDatabase().getRezeptur().dbGetRezeptur(id, true);
    			if(ri != null) {
    				ret = getLosnummerVorschlag1(ri.getTitel(),ri.getLosnummer(),ri.getArtikelnummer());
    			}
    		}else {
    	    	 ret = getLosnummerVorschlag1(mi.getHeader().getBezeichnung(),"","");    			
    		}
    	 }
    	 return ret;
     }

     
    private String getLosnummerVorschlag1(String name,String losnummer,String artikelnummer) {
         String ret = "";
         final int nextNumber=0;
         String template="";
         final Date now = db.getServerTimestamp();
         final SimpleDateFormat monat = new SimpleDateFormat("MM");
         final SimpleDateFormat jahr2 = new SimpleDateFormat("yy");
         final SimpleDateFormat jahr4 = new SimpleDateFormat("yyyy");
         final SimpleDateFormat tag=new SimpleDateFormat("dd");
         final SimpleDateFormat stunde= new SimpleDateFormat("HH");
         final SimpleDateFormat minute = new SimpleDateFormat("mm");
         final SimpleDateFormat julian = new SimpleDateFormat("D");
         final SimpleDateFormat sekunde = new SimpleDateFormat("ss");
         
         template = getLosnummerVorschlagTemplate();
         /* templates ersetzen
       	  * $w = Tag des Jahres (Dezimal)
       	  * $W = Tag des Jahres (DuoDezimal)
          * $m = Monat
          * $d = Tag der Herstellung
          * $y = Jahr (2-stellig)
          * $Y = Jahr (4-stellig)
          * $h = Stunde
          * $M = minute
          * $l = losnummer-vorlage aus rezeptur
          * $a = Artikelnummer
          * $u = Benutzer (vollständig)
          * $U = Benutzer (nur 1 Buchstabe)
          * $b = Anfangsbuchstaben der Rezeptur
          * $4 = die ersten 4 Buchstaben der Rezeptur
          * $s = Sekunde
         */
         String ab ="";
         String ab4 = "";
         try {
	         if (name.length() >=4) {
	        		 ab4 +=name.substring(0,4);
	         } else {
	        	 ab4 +=name.substring(0,name.length());
	         }
         } catch (final Exception e) {}
         final String worte[] = name.split(" ");
         for (final String element : worte) {
        	 try {
        		 ab +=element.charAt(0);
        	 } catch (final Exception e) { }
         }
         ret = template.replaceAll("\\$n",String.valueOf(nextNumber))
             .replaceAll("\\$w",julian.format(now))
             .replaceAll("\\$W",rz.getDatabase().getKalender().getDate20(Integer.valueOf(julian.format(now))))
         	.replaceAll("\\$m",monat.format(now))
         	.replaceAll("\\$y",jahr2.format(now))
         	.replaceAll("\\$Y",jahr4.format(now))
         	.replaceAll("\\$h",stunde.format(now))
         	.replaceAll("\\$M",minute.format(now))
         	.replaceAll("\\$s",sekunde.format(now))
         	.replaceAll("\\$d",tag.format(now))
         	.replaceAll("\\$l",losnummer.trim())
         	.replaceAll("\\$a",artikelnummer.trim())
         	.replaceAll("\\$b",ab)
         	.replaceAll("\\$4",ab4)
         	.replaceAll("\\$u",db.dbGetUser())
         	.replaceAll("\\$U",String.valueOf(db.dbGetUser().charAt(0)));
         // jetzt noch alle übriggebliebenenen Verbotenen Zeichen entfernen
         ret = rz.getOptionFactory().deleteInvalidChars(ret);
         return ret;
     }

    
    public String getNextLosnummer(RechenItem ri) {
    	return getLosnummer(ri,0);
    }
    
    public String getActualLosnummer(RechenItem ri,int actual) {
    	return getLosnummer(ri,actual);
    }
     /**
      * holt die nächste gültige Losnummer als String
      * @return losnummer
      */
     public String getLosnummer(RechenItem ri,int actual) {
         String ret = "";
         int nextNumber=0;
         String template="";
         final Date now = db.getServerTimestamp();
         final SimpleDateFormat monat = new SimpleDateFormat("MM");
         final SimpleDateFormat jahr2 = new SimpleDateFormat("yy");
         final SimpleDateFormat jahr4 = new SimpleDateFormat("yyyy");
         final SimpleDateFormat tag=new SimpleDateFormat("dd");
         final SimpleDateFormat stunde= new SimpleDateFormat("H");
         final SimpleDateFormat minute = new SimpleDateFormat("mm");
         final SimpleDateFormat julian = new SimpleDateFormat("D");
         try {
             ResultSet rs = null;
             // erst die nächste Losnummer ermitteln
             if(actual <= 0) {
	             rs = getNextLosnummer.executeQuery();
	             if(rs.next()) {
	                 nextNumber = rs.getInt("max")+1;
	             }
	             rs.close();
             } else {
            	 nextNumber = actual;
             }
             // dann das Losnummer template holen
             template = getLosnummerTemplate();
             /* templates ersetzen
              * $w = Tag des Jahres (Dezimal)
              * $W = Tag des Jahres (DuoDezimal)
              * $n = nächste Losnummer
              * $m = Monat
              * $d = Tag der Herstellung
              * $y = Jahr (2-stellig)
              * $Y = Jahr (4-stellig)
              * $h = Stunde
              * $M = minute
              * $l = losnummer-vorlage aus rezeptur
              * $a = Artikelnummer
              * $u = Benutzer (vollständig)
              * $U = Benutzer (nur 1 Buchstabe)
              * $x = nächste Losnummer - hexadezimal
              *
             */
             ret = template.replaceAll("\\$n",String.valueOf(nextNumber))
            	.replaceAll("\\$w",julian.format(now))
            	.replaceAll("\\$W",rz.getDatabase().getKalender().getDate20(Integer.valueOf(julian.format(now))))
             	.replaceAll("\\$m",monat.format(now))
             	.replaceAll("\\$y",jahr2.format(now))
             	.replaceAll("\\$Y",jahr4.format(now))
             	.replaceAll("\\$h",stunde.format(now))
             	.replaceAll("\\$M",minute.format(now))
             	.replaceAll("\\$d",tag.format(now))
             	.replaceAll("\\$l",ri.getRezeptur().getLosnummer().trim())
             	.replaceAll("\\$a",ri.getRezeptur().getArtikelnummer().trim())
             	.replaceAll("\\$u",db.dbGetUser())
             	.replaceAll("\\$U",String.valueOf(db.dbGetUser().charAt(0)))
             	.replaceAll("\\$x", Integer.toHexString(nextNumber).toLowerCase());
             // jetzt noch alle übriggebliebenenen Verbotenen Zeichen entfernen
             ret = rz.getOptionFactory().deleteInvalidChars(ret);
             boolean ok = false;
          	String l = ret;
             while (!ok) {
             	if(this.checkLosnummer(l) == true) {
             		// Losnummer existiert, gültige Losnummer erfragen
                    final DialogItem di  = new DialogItem(
                            DialogControl.EINGABE_DIALOG,
                            "",0.0,
                            rz.getLocale().getString("buchung.error_losnummer"),
                            rz.getLocale().getString("buchung.error_losnummer_label"),
                            l,
                            "frage.png",
                            rz.getLocale().getString("string_ok"),
                            rz.getLocale().getString("string_abbrechen"),
                            null
                    );
                    rz.getDialogFactory().getDialog(di);
                    if(di.getReplyCode() == DialogControl.DIALOG_OK) {
                    	l = di.getContent();
                    } else {
                    	ret=null;
                    	ok=true;
                    }
             	} else {
             		ret = l;
             		ok=true;
             	}
             }
             if(rs != null) {
                 rs.close();
             }
         } catch(final Exception e) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".getNextLosNummer", e.getLocalizedMessage());
	         ret = null;
         }
         return ret;
     }

     /**
      * plaziert eine Herstellung in der Datenbank, durch einfügen in die Tabelle der Herstellungen
      * Diese Funktion ist _immer_ Teil einer Transaction, zu der zur Herstellungsbuchung selbst noch
      * mehrere Buchungen in der tabelle "buchungen" gehören
      * @param ri
      * @param losnummer
      * @param typ
      * @param schwundsatz
      * @param endmenge
      * @return 0, wenn ein Fehler aufgetreten ist, !=0 ist der Primärschlüssel des Neuen Eintrags in der
      * Tabelle "herstellungen"
      */
     private int  addHerstellung(RechenItem ri,String losnummer,int typ,double schwundsatz,double endmenge,Date datum,String tank,boolean r_probe) {
         int ret = 0;
         try {
         	double liter = ri.getGesamtLiter();
         	double kg = ri.getGesamtKg();
         	double la = ri.getGesamtLA();
            double endLiter=ri.getGesamtLiter();
            double endKg=ri.getGesamtKg();
            double endLa=ri.getGesamtLA();
            if(typ == BUCHUNG_VERSAND) {
            	liter += (ri.getGesamtLiter() * schwundsatz)/100;
            	kg += (ri.getGesamtKg() * schwundsatz)/100;
            	la += (ri.getGesamtLA() * schwundsatz)/100;
            }   else {
            	endLiter = liter -((ri.getGesamtLiter() * schwundsatz)/100);
            	endKg = kg -((ri.getGesamtKg() * schwundsatz)/100);
            	endLa = la - ((ri.getGesamtLA() * schwundsatz)/100);
            }

             saveHerstellung.setInt(1,ri.getRezeptur().getID());
             saveHerstellung.setString(2,ri.getRezeptur().getTitel());
             saveHerstellung.setString(3,db.dbGetDateFormatString(datum));
             saveHerstellung.setString(4, losnummer);
             saveHerstellung.setDouble(5,liter);
             saveHerstellung.setDouble(6,kg);
             saveHerstellung.setDouble(7,la);
             saveHerstellung.setDouble(8,schwundsatz);
             saveHerstellung.setDouble(9,endLiter);
             saveHerstellung.setDouble(10,endKg);
             saveHerstellung.setDouble(11,endLa);
             saveHerstellung.setString(12,db.dbGetUser());
             saveHerstellung.setString(13,tank);
             saveHerstellung.setInt(14,r_probe?1:0);
             if(saveHerstellung.executeUpdate() != 0) {
                 ret = db.dbGetLastInsertID(DBFactory.TABLE_BUCHUNG);
             } else {
                 ret = 0;
             }
         } catch(final Exception e) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".addHerstellung", e.getLocalizedMessage());
	         ret = 0;
         }
         return ret;
     }

     /**
      * bucht eine Herstellung (=Herstellbuchung in der Tabelle "herstellungen", 1 Zugang in der tabelle "buchungen" und
      * x Abgänge in der Tabelle "buchungen") . wird aus einer Transaction aufgerufen.
      * @param ri
      * @param typ
      * @param schwundsatz
      * @param endmenge
      * @return true, wenn alle Buchungen vorgenommen werden konnten, false, wenn ein Fehler aufgetreten ist und
      * keine der Buchungen geschrieben werden konnte.
      */
     private herstellungReturnItem herstellungBuchen(RechenItem ri,int typ,double schwundsatz,double endmenge,String bemerkungen,String tank,boolean r_probe) {
         herstellungReturnItem ret=new herstellungReturnItem(false,"");
         // Alle Buchungen werden mit demselben Datumsstempel ausgeführt !
         final Date datum = db.getServerTimestamp();
         final BuchungsListe abgaenge = new BuchungsListe();
         String losnummer = getNextLosnummer(ri);
         if(losnummer != null) {
        	 ret.setLosnummer(losnummer);
	         double endLiter=ri.getGesamtLiter();
	         double endKg=ri.getGesamtKg();
	         double endLa=ri.getGesamtLA();
	         if(typ == BUCHUNG_VERSAND) {
	         	endLiter += (ri.getGesamtLiter() * schwundsatz)/100;
	         	endKg += (ri.getGesamtKg() * schwundsatz)/100;
	         	endLa += (ri.getGesamtLA() * schwundsatz)/100;
	         }
	         int i;
	         final int herstellung = addHerstellung(ri,losnummer,typ,schwundsatz,endmenge,datum,tank,r_probe);
	         if(herstellung != 0) {
	        	 String losnummer1 = getActualLosnummer(ri,herstellung);
	        	 if(!losnummer1.equals(losnummer)) {
	        		 losnummer = losnummer1;
		        	 try {
		        		 setHerstellungLosnummer.setString(1,losnummer);
		        		 setHerstellungLosnummer.setInt(2, herstellung);
		        		 setHerstellungLosnummer.executeUpdate();
		        	 }catch(Exception e) { }
	        	 } else {
	        	 }
	        	 ret.setLosnummer(losnummer);
		         if(!losnummer.equals("")) {
		        	 	SteuerDatenItem si = rz.getDatabase().getRezeptur().getSteuerDaten(ri.getRezeptur().getID());
		             final BuchungsItem zugang = new BuchungsItem(
		                     rz,
		                     0,
		                     TYP_ZUGANG,
		                     CODE_HERSTELLUNG,
							 ri.getRezeptur().getLager(),
		                     herstellung,
		                     ri.getRezeptur().getID(),
		                     null,// ri.getRezeptur().getID() == 0 ? null : db.getRezeptur().dbGetRezeptur(ri.getRezeptur().getID(),true),
		                     datum,
		                     db.dbGetUser(),
		                     ri.getRezeptur().getTitel(),
		                     losnummer,
		                     REIHE_FIFO,
		                     (tank != null && !tank.equals("")) ? "[Tank " + tank + "] " +bemerkungen : bemerkungen,
		                     false,
		                     false,
		                     ri.getRezeptur().getBestandseinheit(),
		                     ri.getStaerke(),
		                     ri.getLitergewicht(),
		                     endLiter,
		                     endKg,
		                     endLa,
		                     endLiter,
		                     endKg,
		                     endLa,
		                     true,
		                     true,
		     	             0,
		     	             0,
                             si.getSteuer_artikel(),
                             si.getSorte(),
                             si.getSteuerlager(),
                             si.getStrSorte(),
                             si.getStrSteuerlager()
                             
		             );
		             // stärke und Litergewicht neu setzen, weil das Zugangs-BuchungsItem diese Werte
		             // aus der Rezeptur bezogen hat
		             zugang.setStaerke(ri.getStaerke());
		             zugang.setLitergewicht(ri.getGesamtLitergewicht());
		             zugang.setBezeichnung(ri.getRezeptur().getTitel());
		             if(zugang != null) {
		                 if(zugangBuchen(zugang)) {
		                     for(i = 0;i < ri.getZeilenListe().size();i++) {
		                         if(ri.getZeilenListe().getItem(i).isRechnen()) {
			                         if(ri.getZeilenListe().getItem(i).getRezeptur().getID() != 0) {
		                                endLiter = ri.getZeilenListe().getItem(i).getLiter();
		                                endKg = ri.getZeilenListe().getItem(i).getKg();
		                                endLa  = ri.getZeilenListe().getItem(i).getLa();
		                                if(typ == BUCHUNG_VERSAND) {
		                                 	endLiter += (ri.getZeilenListe().getItem(i).getLiter() * schwundsatz)/100;
		                                 	endKg += (ri.getZeilenListe().getItem(i).getKg()  * schwundsatz)/100;
		                                 	endLa += (ri.getZeilenListe().getItem(i).getLa()  * schwundsatz)/100;
		                                 }
		                                SteuerDatenItem si1 = rz.getDatabase().getRezeptur().getSteuerDaten(ri.getRezeptur().getID());
				                         abgaenge.add( new BuchungsItem(
				                                 rz,
				                                 0,
				                                 TYP_ABGANG,
				                                 CODE_HERSTELLUNG,
												 ri.getZeilenListe().getItem(i).getRezeptur().getLager(),
				                                 herstellung,
				                                 ri.getZeilenListe().getItem(i).getRezeptur().getID(),
				                                 null,//ri.getZeilenListe().getItem(i).getRezeptur().getID()  == 0 ? null : db.getRezeptur().dbGetRezeptur(ri.getZeilenListe().getItem(i).getRezeptur().getID(),true),
				                                 datum,
				                                 db.dbGetUser(),
				                                 ri.getZeilenListe().getItem(i).getBezeichnung(),
				                                 losnummer,
				                                 ri.getZeilenListe().getItem(i).getReihenfolge(),
				                                 "",
				                                 false,
				                                 false,
				                                 ri.getZeilenListe().getItem(i).getRezeptur().getBestandseinheit(),
				                                 ri.getZeilenListe().getItem(i).getStaerke(),
				                                 ri.getZeilenListe().getItem(i).getLitergewicht(),
												 -endLiter,
												 -endKg,
												 -endLa,
				                                 0,0,0,
				                                 false,
				                                 true,
				                                 0,
				                                 0,
				                                 si1.getSteuer_artikel(),
				                                 si1.getSorte(),
				                                 si1.getSteuerlager(),
				                                 si1.getStrSorte(),
				                                 si1.getStrSteuerlager()
				                         ));
			                         }
		                         }
		                     }
		                     /* */
		                     for(i = 0 ; i< abgaenge.size();i++) {
		                     	ret.setRet(abgangBuchen(abgaenge.getItem(i)));
		                     	if(!ret.isRet()) {
		                     		break;
		                     	}
		                     }
		                 }
		             }
	             }
	         }
         }
         return ret;
     }
     /**
      * Transaction zum Einfügen einer Herstellung (Herstellbuchung, Zu/Abgangsbuchungen) in die Datenbank
      * @param ri
      * @param typ
      * @param schwundsatz
      * @param endmenge
      * @return true, wenn die Herstellung komplett gebucht wurde, false, wenn ein Fehler aufgetreten ist.
      */
     public herstellungReturnItem herstellungBuchenTransaction(RechenItem ri,int typ,double schwundsatz,double endmenge,String bemerkungen,String tank,boolean r_probe) {
         herstellungReturnItem ret = new herstellungReturnItem(false,"");
         try {
             con.setAutoCommit(false);
             herstellungReturnItem r  =herstellungBuchen(ri,typ,schwundsatz,endmenge,bemerkungen,tank,r_probe);
             if(!r.isRet()) {
                 con.rollback();
                 ret.setRet(false);
             }
             ret.setRet(true);
             ret.setLosnummer(r.getLosnummer());
             con.setAutoCommit(true);
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".herstellungBuchenTransaction.rollback", ex.getLocalizedMessage());
	         ret.setRet(false);
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + ".herstellungBuchenTransaction.2", ex.getLocalizedMessage());
			                ret.setRet(false);
			            }
		            }
	         }
	     if (!ret.isRet()) {
	         failMessage();
	     }
         return ret;
     }

     /**
      * BuchungsItem stornieren.... nach einem Test mit checkStornoOK
      * @param bi
      * @return true, wenn storniert werden konnte
      */
     private boolean buchungStornieren(BuchungsItem bi) {
         boolean ret = false;
         try {
             // bestand ermitteln und buchen
             double bestand;
             RezepturItem ri = db.getRezeptur().dbGetRezeptur(bi.getRezeptur_id(),true);
             if(ri.getVerweis() !=0) {
				ri = db.getRezeptur().dbGetRezeptur(ri.getVerweis(),true);
			}
             bestand = ri.getBestand();
             double value = 0;
             switch (ri.getBestandseinheit()) {
             	case RzPro.EINHEIT_LITER: value = bi.getLiter(); break;
             	case RzPro.EINHEIT_KG: value = bi.getKg(); break;
             	case RzPro.EINHEIT_LA: value = bi.getLa(); break;
             }
             bestand -=value; // bestand zurückbuchen
             if(db.getRezeptur().updateBestand(bi.getRezeptur_id(),bestand)){
                 boolean ok = false;
                 // bestand korrigiert
                 // bei Entnahme muss auch noch die zugangsbuchung korrigiert werden
                 if(bi.getTyp()==TYP_ABGANG) {
	                 getZugangsBuchung.setInt(1,bi.getRezeptur_id());
	                 getZugangsBuchung.setString(2,bi.getLosnummer());
	                 final ResultSet rs = getZugangsBuchung.executeQuery();
	                 if(rs.next()) {
	                     int rest = 0;
	                     final double liter=rs.getDouble("rest_liter") + Math.abs(bi.getLiter());
	                     final double kg = rs.getDouble("rest_kg") + Math.abs(bi.getKg());
	                     final double la = rs.getDouble("rest_la") + Math.abs(bi.getLa());
	                     switch(ri.getBestandseinheit()) {
	                     	case RzPro.EINHEIT_LITER :
	                     	    if(liter >= precision) {
	                     	        rest = 1;
	                     	    } else {
	                     	        rest = 0;
	                     	    }
	                     		break;
	                     	case RzPro.EINHEIT_KG :
	                     	    if(kg >=  precision) {
	                     	        rest = 1;
	                     	    } else {
	                     	        rest = 0;
	                     	    }
	                     		break;
	                     	case RzPro.EINHEIT_LA :
	                     	    if(la >=  precision) {
	                     	        rest = 1;
	                     	    } else {
	                     	        rest = 0;
	                     	    }
	                     		break;
	                     }
	                     stornoEntnahme.setDouble(1,liter);
	                     stornoEntnahme.setDouble(2,kg);
	                     stornoEntnahme.setDouble(3,la);
	                     stornoEntnahme.setInt(4,rest);
	                     stornoEntnahme.setInt(5,rs.getInt("buchungen.id"));
	                     if(stornoEntnahme.executeUpdate() != 0) {
	                         ok=true;
	                     } else {
	                         ok=false;
	                     }
	                     ok=true;
	                 } else {
	                     ok=false;
	                 }
                 } else {
                     ok=true;
                 }
                 // jetzt noch die Buchung selbst löschen
                 if(ok) {
	                 deleteBuchung.setInt(1,bi.getId());
	                 if(deleteBuchung.executeUpdate() != 0) {
	                     ret = true;
	                 }
                 } else {
                     ret =false;
                 }
             }
         }  catch (final Exception e){
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.buchungStornieren", e.getLocalizedMessage());
             ret = false;
         }
         return ret;
     }
     /**
      * prüfen, ob nachfolgende Buchungen für ein bestimmtes BuchungsItem existieren
      * @param bi
      * @return  true, wenn keine nachfolgenden buchungen existieren, wenn also eine
      * Buchung storniert werden kann
      */
     private boolean checkStornoOK(BuchungsItem bi) {
         boolean ret = false;
         try {
             checkLastEntry.setInt(1,bi.getRezeptur_id());
             checkLastEntry.setString(2,db.dbGetDateFormatString(bi.getDatum()));
             final ResultSet rs = checkLastEntry.executeQuery();
             ret = !rs.next();
             if(rs != null) {
                 rs.close();
             }
         }  catch (final Exception e){
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.checkStornoOK", e.getLocalizedMessage());
             ret = false;
         }
         return ret;
     }

     /**
      * buchung prüfen ob sie storniert werden kann und sie ggfs stornieren
      * @param bi  buchungsitem
      * @return true, wenn gelöscht wurde
      */
     private boolean storno(BuchungsItem bi) {
     	boolean ret = false;
     	if(checkStornoOK(bi)) {
     		ret = buchungStornieren(bi);
     	}
     	return ret;
     }

     /**
      * einen zugang oder abgang stornieren... nur für einzelbuchungen, nicht für herstellungen !
      * @param bi   buchungsitem zum stornieren
      * @return
      */
     public boolean stornoTransaction(BuchungsItem bi) {
         boolean ret =true;
         try {
             con.setAutoCommit(false);
             ret = storno(bi);
             if(ret == false){;
             	con.rollback();
             }
             con.setAutoCommit(true);
             if(bi.getRezeptur_id() != 0) {
            	 rz.getDatabase().getChanges().cleanupAfterStorno(bi.getRezeptur_id());
             }
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + "stornoTransaction.rollback", ex.getLocalizedMessage());
	         ret = false;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_BUCHUNG + "stornoTransaction.2", ex.getLocalizedMessage());
			                ret = false;
			            }
		            }
	         }
	     if(ret == false ) {
	         failStornoMessage();
	     }
         return ret;
     }

     /**
      * invertiert das erledigen flag einer herstellung und der zugehörigen buchungen
      * @param hi   herstellungsItem
      * @return
      */
     public boolean erledigenTransaction(HerstellungsItem hi) {
         boolean ret =false;
         try {
             con.setAutoCommit(false);
             hi.setErledigt(!hi.isErledigt());
             setHerstellungErledigt.setBoolean(1,hi.isErledigt());
             setHerstellungErledigt.setInt(2,hi.getId());
             if(setHerstellungErledigt.executeUpdate() != 0) {
             	setBuchungenErledigt.setBoolean(1,hi.isErledigt());
             	setBuchungenErledigt.setInt(2,hi.getId());
             	if(setBuchungenErledigt.executeUpdate() != 0) {
             		ret = true;
             	} else {
             		ret = false;
             	}
             }
             if(ret == false){;
             	con.rollback();
             	hi.setErledigt(!hi.isErledigt());
             }
             con.setAutoCommit(true);
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_HERSTELLUNG + "erledigenTransaction.rollback", ex.getLocalizedMessage());
	         ret = false;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_HERSTELLUNG + "erledigenTransaction.2", ex.getLocalizedMessage());
			                ret = false;
			            }
		            }
	         }
	     if(ret == false ) {
	     	failErledigtMessage();
	     }
         return ret;
     }

     /**
      * setzt die Felder einer Buchung, die auch nach der Buchung noch bearbeitet werden können
      * (bezeichung,lager,bemerkungen
      * @param bi
      * @return true, wenn erfolgreich
      */
     private boolean updateEinzelBuchung(BuchungsItem bi) {
     	boolean ret = false;
        try {
        	updateBuchung.setString(1,bi.getBezeichnung());
        	updateBuchung.setString(2,bi.getComment());
        	updateBuchung.setInt(3,bi.getLager());
        	updateBuchung.setString(4, bi.getSteuer_artikel());
        	updateBuchung.setInt(5,bi.getSorte());        	
        	updateBuchung.setInt(6,bi.getSteuerlager());
        	updateBuchung.setInt(7,bi.getId());
        	if(updateBuchung.executeUpdate() != 0) {
        		ret = true;
        	}
         } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.updateEinzelBuchung", e.getLocalizedMessage());
             ret = false;
         }
        return ret;
     }

     /**
      * setzt die Felder einer Herstellung, die auch nach der Buchung noch verändert werden können
      * (Bezeichnung,schwundsatz,endmenge)
      * Hinweis: Bemerkungen finden sich in der zur Herstellung gehörenden Zugangsbuchung
      * @param hi
      * @return
      */
     private boolean updateEinzelHerstellung(HerstellungsItem hi) {
     	boolean ret = false;
        try {
        	updateHerstellung.setString(1,hi.getBezeichnung());
        	updateHerstellung.setDouble(2,hi.getSchwundsatz());
        	updateHerstellung.setDouble(3,hi.getEndmenge_liter());
        	updateHerstellung.setDouble(4,hi.getEndmenge_kg());
        	updateHerstellung.setDouble(5,hi.getEndmenge_la());
        	updateHerstellung.setString(6,hi.getTank()); 
        	updateHerstellung.setInt(7,hi.isRProbe()==true?1:0);
        	updateHerstellung.setInt(8,hi.getId());
        	if(updateHerstellung.executeUpdate() != 0) {
        		ret = true;
        	}
         } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.updateEinzelherstellung", e.getLocalizedMessage());
             ret = false;
         }
        return ret;
     }

     /**
      * versucht, eine Herstellung wenn möglich aus der Datenbank zu löschen. Ist Teil einer größeren
      * Transaktion, die auch die dazugehörenden Zugangs-/Abgangsbuchungen zu stornieren versucht.
      * @param hi
      * @return true, wenn die Herstellung entfernt werden konnte.
      */
     private boolean deleteEinzelHerstellung(HerstellungsItem hi) {
     	boolean ret = false;
        try {
        	ResultSet rs;
        	checkLastHerstellung.setInt(1,hi.getRezeptur());
        	checkLastHerstellung.setString(2,db.dbGetDateFormatString(hi.getDatum()));
        	rs = checkLastHerstellung.executeQuery();
        	if(!rs.first()) {
        		// herstellung ist der letzte Vorgang für die Rezeptur
        		deleteHerstellung.setInt(1,hi.getId());
        		if(deleteHerstellung.executeUpdate() != 0) {
        			ret = true;
        		}
        	}
        	if(rs != null) {
        		rs.close();
        	}
         } catch(final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBuchung.deleteEinzelherstellung", e.getLocalizedMessage());
             ret = false;
         }
        return ret;
     }

     /**
      * speichert eine geänderte Herstellung und deren Buchungen
      * @param hi   herstellungsItem
      * @param bl   buchungsliste;
      * @return
      */
     public boolean  updateHerstellungTransaction(HerstellungsItem hi,BuchungsListe bl) {
         boolean ret =false;
         try {
             con.setAutoCommit(false);
             if(updateEinzelHerstellung(hi)) {
             	for(int i = 0; i < bl.size();i++) {
             		ret = updateEinzelBuchung(bl.getItem(i));
             		if(ret == false) {
             			break;
             		}
             	}
             }
             if(ret == false){;
             	con.rollback();
             }
             con.setAutoCommit(true);
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_HERSTELLUNG + "editHerstellungTransaction.rollback", ex.getLocalizedMessage());
	         ret = false;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_HERSTELLUNG + "editHerstellungTransaction.2", ex.getLocalizedMessage());
			                ret = false;
			            }
		            }
	         }
	     if(ret == false ) {
	     	failMessage();
	     }
         return ret;
     }

     /**
      * löscht (wenn möglich) eine herstellung. Entfernt zunächst die Herstellungsbuchung und versucht danach
      * Schritt für Schritt die entsprechenden Zugangs-/Abgangsbuchungen zu stornieren
      * @param hi   herstellungsItem
      * @param bLetzte   buchungsliste;
      * @return true, wenn alle Buchungen storniert werden konnten und die Herstellung danach gelöscht wurde.
      */
     public boolean  deleteHerstellungTransaction(HerstellungsItem hi) {
         boolean ret =false;
         try {
         	final BuchungsListe bl = getHerstellung(hi.getId());
         	if(bl != null) {
	             con.setAutoCommit(false);
	             if(deleteEinzelHerstellung(hi)) {
	             	for(int i = 0;i < bl.size();i++) {
	             		ret = storno(bl.getItem(i));
	             		if(ret == false) {
	             			dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
	             	        dlgItem.setCaption(rz.getLocale().getString("database.buchung.storno_fail"));
	             	        dlgItem.setTitel(rz.getLocale().getString("database.buchung.storno_fail"));
	             	        dlgItem.setMessage(rz.getLocale().getString("database.buchung.storno_fail_message").replace("%s", bl.getItem(i).getBezeichnung()));
	             	        dlgItem.setIcon("warnung.png");
	             	        dlgItem.setOk(rz.getLocale().getString("string_ok"));
	             	        rz.getDialogFactory().getDialog(dlgItem);
	             			break;
	             		}
	             	}
	             }
	             if(ret == false){;
	             	con.rollback();
	             }
	             con.setAutoCommit(true);
	             if(hi.getRezeptur() != 0) {
	            	 rz.getDatabase().getChanges().cleanupAfterStorno(hi.getRezeptur());
	             }
         	}
	     } catch(final Exception ex) {
	         rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_HERSTELLUNG + "deleteHerstellungTransaction.rollback", ex.getLocalizedMessage());
	         ret = false;
	         if (con != null) {
		            try {
			            con.rollback();
			            con.setAutoCommit(true);
			            } catch(final Exception excep) {
			                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DB" + DBFactory.TABLE_HERSTELLUNG + "deleteHerstellungTransaction.2", ex.getLocalizedMessage());
			                ret = false;
			            }
		            }
	         }
	     if(ret == false ) {
	     	failDeleteHerstellungMessage();
	     }
         return ret;
     }


     /**
      * zeigt die Fehlermeldung an, wenn das Löschen einer Herstellung unmöglich ist
      *
      */
     private void failDeleteHerstellungMessage() {
         dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
         dlgItem.setCaption(rz.getLocale().getString("database.buchung.delete_fail"));
         dlgItem.setTitel(rz.getLocale().getString("database.buchung.delete_fail"));
         dlgItem.setMessage(rz.getLocale().getString("database.buchung.delete_fail_message"));
         dlgItem.setIcon("warnung.png");
         dlgItem.setOk(rz.getLocale().getString("string_ok"));
         rz.getDialogFactory().getDialog(dlgItem);
     }

     /**
      * zeigt die Fehlermeldung an, wenn eine transaktion fehlgeschlagen ist
      *
      */
     private void failErledigtMessage() {
         dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
         dlgItem.setCaption(rz.getLocale().getString("database.buchung.erledigen_fail"));
         dlgItem.setTitel(rz.getLocale().getString("database.buchung.erledigen_fail"));
         dlgItem.setMessage(rz.getLocale().getString("database.buchung.erledigen_fail_message"));
         dlgItem.setIcon("warnung.png");
         dlgItem.setOk(rz.getLocale().getString("string_ok"));
         rz.getDialogFactory().getDialog(dlgItem);
     }

     /**
      * zeigt die Fehlermeldung an, wenn eine transaktion fehlgeschlagen ist
      *
      */
     private void failStornoMessage() {
         dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
         dlgItem.setCaption(rz.getLocale().getString("database.buchung.storno_fail"));
         dlgItem.setTitel(rz.getLocale().getString("database.buchung.storno_fail"));
         dlgItem.setMessage(rz.getLocale().getString("database.buchung.storno_fail_message"));
         dlgItem.setIcon("warnung.png");
         dlgItem.setOk(rz.getLocale().getString("string_ok"));
     }

     /**
      * zeigt die Fehlermeldung an, wenn eine transaktion fehlgeschlagen ist
      *
      */
     private void failMessage() {
         dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
         dlgItem.setCaption(rz.getLocale().getString("database.buchung.transaction_fail"));
         dlgItem.setTitel(rz.getLocale().getString("database.buchung.transaction_fail"));
         dlgItem.setMessage(rz.getLocale().getString("database.buchung.transaction_fail_message"));
         dlgItem.setIcon("warnung.png");
         dlgItem.setOk(rz.getLocale().getString("string_ok"));
         rz.getDialogFactory().getDialog(dlgItem);
     }
}
