/*
 *  Copyright (C) database.stichwort_loeschen4 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

// eigene Pakete
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.items.adressen.AdressItem;
import de.lunqual.rzpro.items.bestellung.BestellungsCheckItem;
import de.lunqual.rzpro.items.bestellung.BestellungsItem;
import de.lunqual.rzpro.items.bestellung.BestellungsListe;
import de.lunqual.rzpro.items.bestellung.LastOrdersItem;
import de.lunqual.rzpro.items.buchung.BuchungsItem;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.items.rezeptur.RezepturItem;
import de.lunqual.rzpro.log.LogFactory;
import de.lunqual.rzpro.options.OptionFactory;
/**
 *
 * @author  lunqual
 */
public class DBBestellung{

    RzPro           rz;
    Connection      con;
    DBFactory       db;
    DialogItem		dlgItem;

    PreparedStatement   	getBestellung;
    PreparedStatement		getBestellungslisteAlle;
    PreparedStatement		getBestellungslisteOffene;
    PreparedStatement		getBestellungslisteAlleFilter;
    PreparedStatement		getBestellungslisteOffeneFilter;
    PreparedStatement		saveBestellung;
    PreparedStatement		updateBestellung;
    PreparedStatement		deleteBestellung;
    PreparedStatement		checkBestellung;
    PreparedStatement		getBestellungslisteZugang;
    PreparedStatement		updateRest;
    PreparedStatement		getBestellungRezeptur;
    PreparedStatement   	updateBestellt;
    PreparedStatement		getLastOrders;


    public static final int	LISTE_OFFENE=0;
    public static final int	LISTE_ALLE=1;

    /** Creates a new instance of DBBestellung */
    public DBBestellung(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
        dlgItem = new DialogItem(0,"",0.0,"","","","","","",null);
    }

    private void setStatements(){
        try{
            getBestellung   	     	= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_BESTELLUNG + " where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestellungRezeptur     	= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_BESTELLUNG + " where rest > 0.1 AND rezeptur=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestellungslisteAlle	= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_BESTELLUNG + " order by erstellt,bezeichnung",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestellungslisteOffene		= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_BESTELLUNG + " where rest > 0.0001 order by erstellt,bezeichnung",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            saveBestellung 			= con.prepareStatement("INSERT INTO " +DBFactory.TABLE_BESTELLUNG + " (  rezeptur , adresse , bezeichnung ,artikelnummer ,  firma , einheit ,menge ,  rest , comment,losnummer,erstellt ,geaendert , erledigt, user_1 ,user_2, lieferung,lieferung_datum,bestellt,bestellt_user ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateBestellung        =con.prepareStatement("UPDATE " +DBFactory.TABLE_BESTELLUNG + " set rezeptur=? , adresse=? , bezeichnung=? ,artikelnummer=? ,  firma=? , einheit=? ,menge=? ,  rest=? , comment=?,losnummer=?,erstellt=?,geaendert=? ,erledigt=?, user_1=? ,user_2=? , lieferung=?, lieferung_datum=?, bestellt=?,bestellt_user=? where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteBestellung 			= con.prepareStatement("DELETE FROM " +DBFactory.TABLE_BESTELLUNG + " where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            checkBestellung   	     	= con.prepareStatement("SELECT rezeptur,sum(rest) as rest,einheit,lieferung,lieferung_datum FROM " +DBFactory.TABLE_BESTELLUNG + " where rezeptur=? and rest <> 0 group by rezeptur",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestellungslisteZugang    	= con.prepareStatement("SELECT id,rest,einheit FROM " +DBFactory.TABLE_BESTELLUNG + " where rezeptur=? order by erstellt",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            updateRest     		 = con.prepareStatement("UPDATE " +DBFactory.TABLE_BESTELLUNG + " set rest=?,erledigt=?,losnummer=?,user_2=? where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateBestellt        = con.prepareStatement("UPDATE " +DBFactory.TABLE_BESTELLUNG + " set bestellt=?, bestellt_user=? where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getLastOrders =  con.prepareStatement("SELECT bestellung.erstellt,bestellung.rest,bestellung.bezeichnung,bestellung.adresse,adressen.firma,adressen.vorname,adressen.nachname,bestellung.user_1 from bestellung left join adressen on bestellung.adresse=adressen.id where rezeptur=? order by bestellung.id desc limit ?"   ,ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            getBestellungslisteAlleFilter	= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_BESTELLUNG + " WHERE bezeichnung regexp ?  order by erstellt,bezeichnung",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestellungslisteOffeneFilter		= con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_BESTELLUNG + " where rest > 0.0001  and bezeichnung regexp ? order by erstellt,bezeichnung",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);


        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.setStatements", e.getLocalizedMessage());
        }
    }

    public boolean bestellungAbgelaufen(int rezeptur) {
    	if(rezeptur != 0) {
    		try {
    			getBestellungRezeptur.setInt(1, rezeptur);
    			final ResultSet rs = getBestellungRezeptur.executeQuery();
    			while (rs.next()) {
    				if(rs.getInt("lieferung") != 0) {
    					Calendar datum = Calendar.getInstance();
    					datum.setTime(db.getDateFromString(rs.getString("lieferung_datum")));
    					if(datum.before(rz.getToday())) {
    						return true;
    					}
    				}
    			}
        	} catch (final Exception e){
                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.getBestellungRezeptur", e.getLocalizedMessage());
            }
    	}
    	return false;
    }

    
    public void updateBestellt(int id,boolean bestellt,String bestellt_user) {
    	try {
    		updateBestellt.setInt(1,bestellt==true?1:0);
    		updateBestellt.setString(2, bestellt_user);
    		updateBestellt.setInt(3, id);
    		updateBestellt.executeUpdate();
    	} catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.updateBestellt", e.getLocalizedMessage());
        }
    }
    /**
     * den Rest einer Buchung setzen
     */
    public void updateRest(int id,double rest,String losnummer) {
    	try {
    		updateRest.setDouble(1,rest);
    		updateRest.setString(2,db.dbGetDateFormatString(db.getServerTimestamp()) );
    		updateRest.setString(3,losnummer);
    		updateRest.setString(4,db.dbGetUser());
    		updateRest.setInt(5, id);
    		updateRest.executeUpdate();
    	} catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.updateRest", e.getLocalizedMessage());
        }
    }
    /**
     * Einen Wareneingang auf eine Bestellung verbuchen
     *
     */
    public void zugangBestellung(int rezeptur,double zugang_liter,double zugang_kg,double zugang_la,String losnummer) {
    	try {
    		double rest=0.0;
    		//double unterNull=0.0;
    		final BestellungsCheckItem bci = checkBestellung(rezeptur);
    		if(bci != null) {
    			switch(bci.getEinheit()) {
    				case RzPro.EINHEIT_LITER: rest=zugang_liter;break;
    				case RzPro.EINHEIT_KG: rest=zugang_kg;break;
    				case RzPro.EINHEIT_LA: rest=zugang_la;break;
    			}
    			// mindestens eine offene Bestellung liegt vor
	    		getBestellungslisteZugang.setInt(1,rezeptur);
	    		final ResultSet rs= getBestellungslisteZugang.executeQuery();
	    		while (rs.next()){
	    			final double x = rs.getDouble("rest"); // Der Rest der jeweiligen Buchung
	    			final double y = x-rest ; // soviel bleibt übrig
	    			if ((y<=0) || rz.isZero(y)) {
	    				updateRest(rs.getInt("id"),0.0,losnummer);//wenn null, dann auf null setzen
	    				rest = -y;
	    			} else {
	    				updateRest(rs.getInt("id"),y,losnummer); // ansonsten auf das setzen, was übrig ist und keine weiteren Bestellungen verarbeiten
	    				break;
	    			}
	    		}
    		}
    	} catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.zugangBestellung", e.getLocalizedMessage());
        }
    }
    /**
     * prüfen, ob eine Bestellung für eine bestimmte Rezeptur vorliegt
     */
    public BestellungsCheckItem checkBestellung(int rezeptur) {
    	BestellungsCheckItem ret = null;
    	try {
    		checkBestellung.setInt(1, rezeptur);
    		final ResultSet rs=checkBestellung.executeQuery();
    		if(rs.next()) {
    			if(!rz.isZero(rs.getDouble("rest"))) {
    				Calendar datum = Calendar.getInstance();
    				datum.setTime(db.getDateFromString(rs.getString("lieferung_datum")));
	    			ret = new BestellungsCheckItem(
	    					rezeptur,
	    					rs.getDouble("rest"),
	    					rs.getInt("einheit"),
	    					rz.getDatabase().getEinheiten().getGrundeinheit(rs.getInt("einheit")).getName(),
	    					rs.getInt("lieferung") != 0?true:false,
	    					datum
	    			);
	    		}
    		}
    	} catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.checkBestellung", e.getLocalizedMessage());
            ret = null;
        }
    	return ret;
    }
    /**
     * Liste der Bestellungen zurückgeben
     * @param bi
     * @return
     */
    public BestellungsListe getListe(int what,String filter) {
    	BestellungsListe bl = new BestellungsListe();
    	PreparedStatement stm=null;
    	if(!filter.trim().isEmpty()) {
    		filter = filter.replaceAll("\\s+", ".*");
    	}
    	try {
    		switch (what) {
    			case LISTE_OFFENE:
    				if(!filter.trim().isEmpty()) {
    					stm=getBestellungslisteOffeneFilter;
    					stm.setString(1, filter);
    				} else {
    					stm=getBestellungslisteOffene;
    				}
    				break;
    			case LISTE_ALLE:
    				if(!filter.trim().isEmpty()) {
    					stm=getBestellungslisteAlleFilter;
    					stm.setString(1, filter);
    				} else {
    					stm=getBestellungslisteAlle;
    				}
    				break;
    		}
    		if(stm != null) {
	    		final ResultSet rs = stm.executeQuery();
	    		while(rs.next()) {
	    			final BestellungsItem bi = getBestellung(rs.getInt("id"),0,0);
	    			bl.addItem(bi);
	    		}
    		}
    	}  catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.getListe", e.getLocalizedMessage());
            bl = new BestellungsListe();
        }
    	return bl;
    }
    /**
     * Bestellung löschen
     * @param bi
     * @return
     */
    public void deleteBestellung(int id) {
    	try {
    		deleteBestellung.setInt(1, id);
    		deleteBestellung.executeUpdate();
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.deleteBestellung", e.getLocalizedMessage());
        }
    }
    /*
     * Bestellung sichern
     */
    public boolean saveBestellung(BestellungsItem bi) {
    	boolean ret = true;
    	Date erstellt=null,geaendert=null,erledigt = null;
    	PreparedStatement stm=null;
    	try {
    		if (bi.getId() == 0) {
    			stm = saveBestellung;
    			erstellt = db.getServerTimestamp();
    			geaendert = db.getServerTimestamp();
    			erledigt = db.getServerTimestamp();
    		}else {
    			stm = updateBestellung;
    			erstellt = bi.getErstellt();
    			geaendert = bi.getGeaendert();
    			erledigt = bi.getErledigt_datum();
    			stm.setInt(20,bi.getId());
    		}
    		if(stm != null) {

    			stm.setInt(1,bi.getRezeptur());
    			stm.setInt(2,bi.getAdresse());
    			stm.setString(3,bi.getBezeichnung());
    			stm.setString(4,bi.getArtikelnummer());
    			stm.setString(5,bi.getFirma());
    			stm.setInt(6,bi.getEinheit());
    			stm.setDouble(7,bi.getMenge());
    			stm.setDouble(8,bi.getRest());
    			stm.setString(9,bi.getComment());
    			stm.setString(10,bi.getLosnummer());
    			stm.setString(11,db.dbGetDateFormatString(erstellt));
    			stm.setString(12,db.dbGetDateFormatString(geaendert));
    			stm.setString(13,db.dbGetDateFormatString(erledigt));
    			stm.setString(14,db.dbGetUser());
    			stm.setString(15,db.dbGetUser());
    			stm.setInt(16, bi.isLiefertermin()?1:0);
    			stm.setString(17, db.dbGetDateFormatString(bi.getLiefertermin_datum().getTime()));
    			stm.setInt(18,bi.isBestellt()?1:0);
    			stm.setString(19, bi.getBestellt_user());
                final int r = stm.executeUpdate();
                if(r == 0) {
                    ret = false;
                }
    		}
    	}  catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.saveBestellung", e.getLocalizedMessage());
            ret = false;
        }
    	return ret;
    }

    
    public BestellungsItem getBestellung(int id,int rezeptur,int adresse) {
    	BestellungsItem ret=null;
    	AdressItem ai = null;
    	String adresseText = "";
    	String rezepturText ="";
    	String artikelnummerText="";
    	RezepturItem ri = null;
    	int einheit = RzPro.EINHEIT_LITER;
        try{
           if( id == 0) {
        	   ri = rz.getDatabase().getRezeptur().dbGetRezeptur(rezeptur,true);
        	   if(ri != null) {
        		   einheit = ri.getBasiseinheit();
        	   }
        	   //Neue Bestellung
        	   if (adresse != 0) {
        		   ai = rz.getDatabase().getAdressen().dbGetAdresse(adresse, true);
        		   adresseText = ai.getFirma() + ", " + ai.getVorname()  + " " + ai.getNachname() + ai.getOrt();
        	   }
        	   if (rezeptur != 0) {
        		   rezepturText = rz.getDatabase().getRezeptur().dbGetRezepturName(rezeptur);
        		   artikelnummerText = rz.getDatabase().getRezeptur().getArtikelnummer(rezeptur);
        	   }
        	   ret = new BestellungsItem(rz,0,
        			   rezeptur,
        			   adresse,
        			   rezepturText,
        			   artikelnummerText,
        			   adresseText,
        			   einheit,
        			   0.0,
        			   0.0,
        			   "",
        			   "",
        			   rz.getDatabase().getServerTimestamp(),
        			   rz.getDatabase().getServerTimestamp(),
        			   rz.getDatabase().getServerTimestamp(),
        			   rz.getDatabase().dbGetUser(),
        			   rz.getDatabase().dbGetUser(),
        			   false,
        			   Calendar.getInstance(),
        			   false,
        			   ""
        	);

           }else {
   	    		getBestellung.setInt(1,id);
	    		final ResultSet rs = getBestellung.executeQuery();
	    		if(rs.next()) {
		    		Calendar datum = Calendar.getInstance();
		    		datum.setTime(db.getDateFromString(rs.getString("lieferung_datum")));
	    			ret = new BestellungsItem(rz,id,
	    					rs.getInt("rezeptur") ,
	    	    			rs.getInt("adresse"),
	    	    			rs.getString("bezeichnung"),
	    	    			rs.getString("artikelnummer"),
	    	    			rs.getString("firma"),
	    	    			rs.getInt("einheit"),
	    	    			rs.getDouble("menge"),
	    					rs.getDouble("rest"),
	    	    			rs.getString("comment"),
	    	    			rs.getString("losnummer"),
	    					rs.getDate("erstellt"),
	    					rs.getDate("geaendert"),
	    					rs.getDate("erledigt"),
	    					rs.getString("user_1"),
	    					rs.getString("user_2"),
	    					rs.getInt("lieferung") != 0?true:false,
	    					datum,
	    					rs.getInt("bestellt") != 0?true:false,
	    					rs.getString("bestellt_user")
	    					);
	    		}
       	   }
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.getBestellung", e.getLocalizedMessage());
        }
        return ret;
    }
    private String getLiterToolTip(BuchungsItem bi) {
		double l01=0.0,l02=0.0,l035=0.0,l05=0.0,l07=0.0,l1=0.0;
		l1=Math.abs(bi.getLiter());
		l07=Math.abs(bi.getLiter()*(1/0.7));
		l05=Math.abs(bi.getLiter()*2);
		l035=Math.abs(bi.getLiter()*(1/0.35));
		l02=Math.abs(bi.getLiter()*(1/0.2));
		l01=Math.abs(bi.getLiter()*10);
		String tt = "<html><table border=\"1\">";
		tt+="<tr><td>" + Double.toString(Math.round(l1)) + "&nbsp * 1 " + rz.getLocale().getString("string_liter") + "</td></tr>";
		tt+="<tr><td>" + Double.toString(Math.round(l07)) + "&nbsp* 0,7 " + rz.getLocale().getString("string_liter") + "</td></tr>";
		tt+="<tr><td>" + Double.toString(Math.round(l05)) + "&nbsp* 0,5 " + rz.getLocale().getString("string_liter") + "</td></tr>";
		tt+="<tr><td>" + Double.toString(Math.round(l035)) + "&nbsp* 0,35 " + rz.getLocale().getString("string_liter") + "</td></tr>";
		tt+="<tr><td>" + Double.toString(Math.round(l02)) + "&nbsp* 0,2 " + rz.getLocale().getString("string_liter") + "</td></tr>";
		tt+="<tr><td>" + Double.toString(Math.round(l01)) + "&nbsp* 0,1 " + rz.getLocale().getString("string_liter") + "</td></tr>";
		tt += "</table></html>";
		return tt;
    }
  
    public ArrayList<LastOrdersItem> getLastOrders(int rezeptur){
    	ArrayList<LastOrdersItem> liste= new ArrayList<LastOrdersItem>();
    	try {
    		if(rezeptur != 0) {
    			getLastOrders.setInt(1, rezeptur);
				getLastOrders.setInt(2, rz.getOptionFactory().getOption("last.orders.count", OptionFactory.LAST_ORDERS_COUNT));
				final ResultSet rs = getLastOrders.executeQuery();
				while(rs.next()) {
					liste.add(new LastOrdersItem(rz,
							rezeptur,
							rs.getString("bestellung.bezeichnung"), 
							rs.getString("adressen.firma"), 
							rs.getString("adressen.vorname"), 
							rs.getString("adressen.nachname"),
							rs.getString("bestellung.user_1"),
							rs.getDate("bestellung.erstellt"),
							rs.getDouble("bestellung.rest") >0.0001?false:true ));
				}
    		}
    	}catch (final Exception e){
    	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBBestellung.getLastOrders", e.getLocalizedMessage());
    	        }
    	return liste;
    }
    
}