/*
 *  Copyright (C) 2004/2005 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import javax.swing.JComboBox;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.items.acid.AcidItem;
import de.lunqual.rzpro.items.acid.AcidListe;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.log.LogFactory;

/**
 *
 * @author  lunqual
 */
public class DBAcid{

    RzPro           		rz;
    Connection      	con;
    DBFactory       	db;
    DialogItem			dlgItem;

    PreparedStatement   getAcidListe;
    PreparedStatement   getAcid;
    PreparedStatement   saveAcid;
    PreparedStatement   updateAcid;
    PreparedStatement   deleteAcid;


    /** Creates a new instance of DBLager */
    public DBAcid(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
    }

    private void setStatements(){
        try{
            getAcidListe    = con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_ACID + " ORDER BY name",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getAcid          = con.prepareStatement("SELECT * FROM " +DBFactory.TABLE_ACID + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            updateAcid		= con.prepareStatement("UPDATE " +DBFactory.TABLE_ACID + " SET name=?,factor=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            saveAcid        = con.prepareStatement("INSERT INTO " +DBFactory.TABLE_ACID + " (name,factor) VALUES(?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteAcid    	= con.prepareStatement("DELETE FROM " +DBFactory.TABLE_ACID + "  WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAcid.setStatements", e.getLocalizedMessage());
        }
    }

    /**
     * Die Liste der Säuren aus der DAtenbank holen
     */
    public AcidListe	getAcidListe() {
    	final AcidListe al=new AcidListe();
        try{
        	final ResultSet rs = getAcidListe.executeQuery();
        	while(rs.next()) {
        		al.addItem(new AcidItem(
        				rs.getInt("id"),
        				rs.getInt("system") == 1?true:false,
        			    rs.getString("name"),
        			    rs.getDouble("factor")
        		));
        	}
        	al.setDefault(1); // Weinsäure ist die Basis der Berechnungen !
        							// Die 1 ist die Datenbank-id derselben (siehe DBFactory->1.1.7_5)
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAcid.getAcidListe", e.getLocalizedMessage());
        }
    	return al;
    }

    /**
     * Ein AcidItem speichern oder updaten
     * @param ai AcidItem
     * @return last_insert_id oder 0, wenn nichts gespeichert/updated werden konnte
     */
    public int saveAcid(AcidItem ai) {
    	int ret =0;
    	try {
    		PreparedStatement stm;
    		if(ai.getId()==0) {
    			stm = saveAcid;
    		} else {
    			stm = updateAcid;
    			stm.setInt(3,ai.getId());
    		}
    		stm.setString(1,ai.getName());
    		stm.setDouble(2,ai.getFactor());
    		stm.executeUpdate();
    		ret = db.dbGetLastInsertID(DBFactory.TABLE_ACID);
    	}
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAcid.saveAcid", e.getLocalizedMessage());
            ret = 0;
        }
    	return ret;
    }

    /**
     * Ein einzelnes AcidItem holen
     * @param id
     * @return AcidItem oder null
     */
    public AcidItem getAcid(int id) {
    	AcidItem ai=null;
    	try {
    		if(id ==0) {
    			ai = new AcidItem(0,false,rz.getLocale().getString("acideingabe.neu"),1.0);
    		} else {
	    		getAcid.setInt(1,id);
	    		final ResultSet rs = getAcid.executeQuery();
	    		if(rs.next()) {
	            	ai = new AcidItem(
	            				rs.getInt("id"),
	            				rs.getInt("system") == 1?true:false,
	            			    rs.getString("name"),
	            			    rs.getDouble("factor")
	            		);
	    		}
    		}
    	}
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAcid.getAcid", e.getLocalizedMessage());
            ai = null;
        }
        return ai;
    }

    /**
     * Ein einzelnes AcidItem löschen
     * Wenn Item.isSystem()==true, kann das Item nicht gelöscht werden !
     * @param buchung_id
     * @return true/false
     */
    public boolean deleteAcid(AcidItem ai) {
    	boolean ret = false;
    	try {
    		if(!ai.isSystem()) {
	    		deleteAcid.setInt(1,ai.getId());
	    		if(deleteAcid.executeUpdate() != 0) {
	    			ret = true;
	    		}
	    	}
    	}
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBAcid.deleteAcid", e.getLocalizedMessage());
            ret = false;
        }
        return ret;
    }
    /**
     * ComboBox mit Säureliste füllen
     * @param cmb
     * @param aDefault
     */
    public void getAcidListeCombo(JComboBox cmb,int aDefault){
        final AcidListe al = getAcidListe();
        cmb.removeAllItems();
        for(int i = 0 ; i < al.size();i++){
            cmb.addItem(al.get(i));
        }
        cmb.setSelectedIndex(aDefault);
    }

    /**
     * liefert das im Rechenfenster eingestellte AcidItem
     * @param cmb
     * @param aDefault
     */
    public AcidItem getDefaultAcid(){
        final AcidListe al = getAcidListe();
        return al.getItem(rz.getOptionFactory().getOption("rechenfenster.umrechnung",0));
    }

    /**
     * Konvertieren von WErten
     */
    public double convert(double value,AcidItem source,AcidItem dest) {
    	double ret = 0.0;
    	if(source.getId() == dest.getId()) {
    		ret = value;
    	} else {
    		ret = value * (1/source.getFactor()) *dest.getFactor();
    	}
    	return ret;
    }
}
